"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const bcrypt_1 = __importDefault(require("bcrypt"));
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const customer_schema_1 = require("../../../utils/schemas/admin/customer-management/customer-schema");
const base_controller_1 = __importDefault(require("../base-controller"));
const customer_service_1 = __importDefault(require("../../../services/admin/customer-management/customer-service"));
const branch_model_1 = __importDefault(require("../../../model/admin/operations/branch-model"));
const mongoose_1 = __importDefault(require("mongoose"));
const lodash_1 = require("lodash");
const customer_vendor_api_1 = require("../../../library/api/customer-vendor-api");
const controller = new base_controller_1.default();
class CustomerController extends base_controller_1.default {
    async findAll(req, res) {
        try {
            const { _id, customerCode, page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
            let query = { _id: { $exists: true } };
            const customerData = await res.locals.customer;
            if (status && status !== '') {
                query.status = { $in: Array.isArray(status) ? status : [status] };
            }
            else {
                query.status = '1';
            }
            if (_id) {
                query._id = new mongoose_1.default.Types.ObjectId(_id);
            }
            if (keyword) {
                const keywordRegex = new RegExp(keyword, 'i');
                query = {
                    $or: [
                        { customerTitle: keywordRegex },
                        { customerEmail: keywordRegex },
                        { customerCode: !isNaN(Number(keyword)) ? Number(keyword) : keywordRegex },
                        { customerPhoneNumber: keywordRegex }
                    ],
                    ...query
                };
            }
            if (customerCode) {
                const customerCodeRegex = new RegExp(customerCode, 'i');
                query = {
                    customerCode: customerCodeRegex,
                    ...query
                };
            }
            const sort = {};
            if (sortby && sortorder) {
                sort[sortby] = sortorder === 'desc' ? -1 : 1;
            }
            const customers = await customer_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: customers,
                totalCount: await customer_service_1.default.getTotalCount(query),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching customers' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = customer_schema_1.customerSchema.safeParse(req.body);
            if (validatedData.success) {
                const { countryId, stateId, cityId, customerType, customerTitleType, customerEmail, customerTitle, customerCountryCode, customerPhoneNumber, customerAlternativePhoneNumber, customerAlternativeCountryCode, dob, customerPassword, address1, address2 } = validatedData.data;
                const user = res.locals.user;
                const customerData = {
                    branchId: user.branchId,
                    countryId: countryId || null,
                    stateId: stateId || null,
                    cityId: cityId || null,
                    customerType,
                    customerTitleType,
                    customerTitle,
                    customerEmail,
                    customerImageUrl: (0, helpers_1.handleFileUpload)(req, null, req.file, 'customerImageUrl', 'customer'),
                    customerCountryCode,
                    customerPhoneNumber,
                    customerAlternativePhoneNumber,
                    customerAlternativeCountryCode,
                    customerPassword: customerPassword ? await bcrypt_1.default.hash(customerPassword, 10) : '',
                    dob,
                    address1,
                    address2,
                    status: '1',
                    createdBy: user._id,
                    createdAt: new Date()
                };
                const newCustomer = await customer_service_1.default.create(customerData);
                if ((0, lodash_1.isEmpty)(newCustomer)) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Something went wrong. Please try again',
                    });
                }
                const retVal = await (0, customer_vendor_api_1.postCustomer)(newCustomer);
                return controller.sendSuccessResponse(res, {
                    requestedData: newCustomer,
                    message: 'Customer created successfully!'
                }, 200, {
                    sourceFromId: newCustomer._id,
                    sourceFrom: task_log_1.adminTaskLog.customerManagement.customer,
                    activity: task_log_1.adminTaskLogActivity.create,
                    activityStatus: task_log_1.adminTaskLogStatus.success
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            if (error && error.errors && error.errors.customerEmail) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        customerEmail: error.errors.customerEmail.message
                    }
                });
            }
            else if (error && error.errors && error.errors.customerCode) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        customerCode: error.errors.customerCode.message
                    }
                });
            }
            else if (error && error.errors && error.errors.customerPhoneNumber) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        customerPhoneNumber: error.errors.customerPhoneNumber.message
                    }
                });
            }
            console.log('error', error);
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while creating customer'
            });
        }
    }
    async findOne(req, res) {
        try {
            const customerId = req.params.id;
            if (customerId) {
                const customer = await customer_service_1.default.findOne(customerId);
                return controller.sendSuccessResponse(res, {
                    requestedData: customer,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Customer Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = customer_schema_1.customerSchema.safeParse(req.body);
            if (validatedData.success) {
                const customerId = req.params.id;
                if (customerId) {
                    const customer = await customer_service_1.default.findOne(customerId);
                    const updatedCustomerData = req.body;
                    const checkBranchIdOrSlug = /^[0-9a-fA-F]{24}$/.test(updatedCustomerData.branchCode);
                    let branchQuery = {};
                    if (checkBranchIdOrSlug) {
                        branchQuery._id = new mongoose_1.default.Types.ObjectId(updatedCustomerData.branchCode);
                    }
                    else {
                        branchQuery.branchCode = updatedCustomerData.branchCode;
                    }
                    const branchDetails = await branch_model_1.default.findOne(branchQuery);
                    if (!branchDetails) {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'The branch code is not found',
                        });
                    }
                    const updatedCustomer = await customer_service_1.default.update(customer._id, {
                        ...updatedCustomerData,
                        branchId: branchDetails._id,
                        countryId: updatedCustomerData.countryId || null,
                        stateId: updatedCustomerData.stateId || null,
                        cityId: updatedCustomerData.cityId || null,
                        customerPassword: updatedCustomerData.customerPassword ? (customer.customerPassword === updatedCustomerData.customerPassword ? customer.customerPassword : await bcrypt_1.default.hash(updatedCustomerData.customerPassword, 10)) : '',
                        customerImageUrl: (0, helpers_1.handleFileUpload)(req, await customer_service_1.default.findOne(customer._id), req.file, 'customerImageUrl', 'customer'),
                        updatedAt: new Date()
                    });
                    console.log('branchDetails', updatedCustomer);
                    if (updatedCustomer) {
                        return controller.sendSuccessResponse(res, {
                            requestedData: updatedCustomer,
                            message: 'Customer updated successfully!'
                        }, 200, {
                            sourceFromId: updatedCustomer._id,
                            sourceFrom: task_log_1.adminTaskLog.customerManagement.customer,
                            activity: task_log_1.adminTaskLogActivity.update,
                            activityStatus: task_log_1.adminTaskLogStatus.success
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'Customer Id not found!',
                        });
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Customer Id not found! Please try again with CustomerId',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating customer'
            });
        }
    }
    async destroy(req, res) {
        try {
            const customerId = req.params.id;
            if (customerId) {
                const customer = await customer_service_1.default.findOne(customerId);
                if (customer) {
                    // await CustomerService.destroy(customerId);
                    // return controller.sendSuccessResponse(res, { message: 'Customer deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant to delete this customer!',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This customer details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Customer Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting customer' });
        }
    }
}
exports.default = new CustomerController();
