"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../base-controller"));
const referral_source_schema_1 = require("../../../utils/schemas/admin/customer-management/referral-source.schema");
const referral_source_service_1 = __importDefault(require("../../../services/admin/customer-management/referral-source.service"));
class ReferralSourceController extends base_controller_1.default {
    constructor(referralSourceService) {
        super();
        this.referralSourceService = referralSourceService || new referral_source_service_1.default();
    }
    async findAll(req, res) {
        const { referralSourceId, itemId = '', page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (referralSourceId) {
            query._id = new mongoose_1.default.Types.ObjectId(itemId);
        }
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { referralSourceTitle: keywordRegex }
                ],
                ...query
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const referralSource = await this.referralSourceService.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return this.sendSuccessResponse(res, {
                requestedData: referralSource.data,
                totalCount: referralSource.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return this.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching referralSource' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = referral_source_schema_1.referralSourceShema.safeParse(req.body);
            if (validatedData.success) {
                const { referralSourceTitle, slug } = validatedData.data;
                const user = res.locals.user;
                const referralSourceData = {
                    referralSourceTitle,
                    slug: (0, helpers_1.slugifyManually)(referralSourceTitle),
                    status: '1',
                    createdBy: user._id,
                    createdAt: new Date()
                };
                const newReferralSource = await this.referralSourceService.create(referralSourceData);
                return this.sendSuccessResponse(res, {
                    requestedData: newReferralSource,
                    message: 'ReferralSource created successfully!'
                }, 200, {
                    sourceFromId: newReferralSource._id,
                    sourceFrom: task_log_1.adminTaskLog.customerManagement.referralSource,
                    activity: task_log_1.adminTaskLogActivity.create,
                    activityStatus: task_log_1.adminTaskLogStatus.success
                });
            }
            else {
                return this.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            if (error && error.errors && error.errors.referralSourceTitle && error.errors.referralSourceTitle.properties) {
                return this.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        referralSourceTitle: error.errors.referralSourceTitle.properties.message
                    }
                });
            }
            return this.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while creating referralSource',
            });
        }
    }
    async findOne(req, res) {
        try {
            const referralSourceId = req.params.id;
            if (referralSourceId) {
                const referralSource = await this.referralSourceService.findOne(referralSourceId);
                return this.sendSuccessResponse(res, {
                    requestedData: referralSource,
                    message: 'Success'
                });
            }
            else {
                return this.sendErrorResponse(res, 200, {
                    message: 'ReferralSource Id not found!',
                });
            }
        }
        catch (error) {
            return this.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = referral_source_schema_1.referralSourceShema.safeParse(req.body);
            if (validatedData.success) {
                const referralSourceId = req.params.id;
                if (referralSourceId) {
                    const updatedReferralSourceData = req.body;
                    const updatedReferralSource = await this.referralSourceService.update(referralSourceId, { ...updatedReferralSourceData, updatedAt: new Date() });
                    if (updatedReferralSource) {
                        return this.sendSuccessResponse(res, {
                            requestedData: updatedReferralSource,
                            message: 'ReferralSource updated successfully!'
                        }, 200, {
                            sourceFromId: updatedReferralSource._id,
                            sourceFrom: task_log_1.adminTaskLog.customerManagement.referralSource,
                            activity: task_log_1.adminTaskLogActivity.update,
                            activityStatus: task_log_1.adminTaskLogStatus.success
                        });
                    }
                    else {
                        return this.sendErrorResponse(res, 200, {
                            message: 'ReferralSource Id not found!',
                        });
                    }
                }
                else {
                    return this.sendErrorResponse(res, 200, {
                        message: 'ReferralSource Id not found! Please try again with ReferralSource id',
                    });
                }
            }
            else {
                return this.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            return this.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating referralSource'
            });
        }
    }
    async destroy(req, res) {
        try {
            const referralSourceId = req.params.id;
            if (referralSourceId) {
                const referralSource = await this.referralSourceService.findOne(referralSourceId);
                if (referralSource) {
                    return this.sendErrorResponse(res, 200, {
                        message: 'You cant delete this referralSource',
                    });
                }
                else {
                    return this.sendErrorResponse(res, 200, {
                        message: 'This referralSource details not found!',
                    });
                }
            }
            else {
                return this.sendErrorResponse(res, 200, {
                    message: 'ReferralSource id not found!',
                });
            }
        }
        catch (error) {
            return this.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting referralSource' });
        }
    }
}
exports.default = ReferralSourceController;
