"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../base-controller"));
const zod_1 = require("zod");
const complaint_service_1 = __importDefault(require("../../../services/admin/inventory-management/complaint-service"));
const complaint_schema_1 = require("../../../utils/schemas/admin/inventory-management/complaint-schema");
const category_model_1 = __importDefault(require("../../../model/admin/inventory-management/category-model"));
const category_config_1 = require("../../../utils/config/admin/inventory-management/category-config");
const category_1 = require("../../../utils/admin/category");
const controller = new base_controller_1.default();
class ComplaintController extends base_controller_1.default {
    async findAll(req, res) {
        const { categoryId, page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { complaintTitle: keywordRegex }
                ],
                ...query
            };
        }
        if (categoryId) {
            // async function fetchAllCategories(categoryIds: any[]): Promise<any[]> {
            //     let queue = [...categoryIds];
            //     const allCategoryIds = new Set([...categoryIds]);
            //     while (queue.length > 0) {
            //         const categoriesData = await CategoryModel.find(
            //             { parentCategory: { $in: queue } },
            //             '_id'
            //         );
            //         const childCategoryIds = categoriesData.map(category => category._id);
            //         if (childCategoryIds.length === 0) {
            //             break;
            //         }
            //         queue = childCategoryIds;
            //         childCategoryIds.forEach(id => allCategoryIds.add(id));
            //     }
            //     return Array.from(allCategoryIds);
            // }
            // const categoryIds = await fetchAllCategories([...new Set([new mongoose.Types.ObjectId(categoryId)])]);
            let categoryBatchIds = [];
            if (categoryId) {
                categoryBatchIds.push(new mongoose_1.default.Types.ObjectId(categoryId));
            }
            const uniqueCategoryIds = [...new Set(categoryBatchIds)];
            var categoriesWithChilleds = await category_model_1.default.aggregate((0, category_config_1.categoriesWithChildrenInArrayAggregation)(uniqueCategoryIds));
            const categoryIds = (0, category_1.extractAllCategoryIds)(categoriesWithChilleds);
            query = {
                ...query, categoryId: { $in: categoryIds }
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const complaint = await complaint_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: complaint,
                totalCount: await complaint_service_1.default.getTotalCount(query),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching complaint' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = await complaint_schema_1.complaintSchema.parseAsync(req.body);
            const { categoryId, complaintTitle, slug, description, status } = validatedData;
            const user = res.locals.user;
            const complaintData = {
                categoryId,
                complaintTitle,
                description,
                slug: (0, helpers_1.slugify)(complaintTitle),
                status: status || 'active',
                createdBy: user._id,
                createdAt: new Date(),
                updatedAt: new Date(),
            };
            console.log(complaintData);
            const newComplaint = await complaint_service_1.default.create(complaintData);
            return controller.sendSuccessResponse(res, {
                requestedData: newComplaint,
                message: 'Complaint created successfully!',
            }, 200, {
                sourceFromId: newComplaint?._id,
                sourceFrom: task_log_1.adminTaskLog.inventoryManagement.complaint,
                activity: task_log_1.adminTaskLogActivity.create,
                activityStatus: task_log_1.adminTaskLogStatus.success,
                activityComment: "Complaint created successfully"
            });
        }
        catch (error) {
            if (error instanceof zod_1.ZodError) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(error.errors),
                });
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'An error occurred while creating the complaint.',
            });
        }
    }
    async findOne(req, res) {
        try {
            const complaintId = req.params.id;
            if (complaintId) {
                const complaint = await complaint_service_1.default.findOne(complaintId);
                return controller.sendSuccessResponse(res, {
                    requestedData: complaint,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Complaint Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = await complaint_schema_1.complaintSchema.safeParseAsync(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 400, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors),
                });
            }
            const complaintId = req.params.id;
            if (complaintId) {
                const updatedComplaintData = req.body;
                const updatedComplaint = await complaint_service_1.default.update(complaintId, { ...updatedComplaintData, updatedAt: new Date() });
                if (updatedComplaint) {
                    return controller.sendSuccessResponse(res, {
                        requestedData: updatedComplaint,
                        message: 'Complaint updated successfully!'
                    }, 200, {
                        sourceFromId: updatedComplaint._id,
                        sourceFrom: task_log_1.adminTaskLog.inventoryManagement.complaint,
                        activity: task_log_1.adminTaskLogActivity.update,
                        activityStatus: task_log_1.adminTaskLogStatus.success,
                        activityComment: "Complaint updated successfully"
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Complaint Id not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Complaint Id not found! Please try again with Complaint id',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating complaint'
            });
        }
    }
    async destroy(req, res) {
        try {
            const complaintId = req.params.id;
            if (complaintId) {
                const complaint = await complaint_service_1.default.findOne(complaintId);
                if (complaint) {
                    // await ComplaintService.destroy(complaintId);
                    // return controller.sendSuccessResponse(res, { message: 'Complaint deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this complaint',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This complaint details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Complaint id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting complaint' });
        }
    }
}
exports.default = new ComplaintController();
