"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const zod_1 = require("zod");
const lodash_1 = require("lodash");
const helpers_1 = require("../../../../utils/helpers");
const task_log_1 = require("../../../../constants/admin/task-log");
const items_master_schema_1 = require("../../../../utils/schemas/admin/inventory-management/items/items-master-schema");
const base_controller_1 = __importDefault(require("../../base-controller"));
const item_details_model_1 = __importDefault(require("../../../../model/admin/inventory-management/items/item-details-model"));
const item_details_service_1 = __importDefault(require("../../../../services/admin/inventory-management/items/item-details-service"));
const branch_model_1 = __importDefault(require("../../../../model/admin/operations/branch-model"));
const stock_inventory_model_1 = __importDefault(require("../../../../model/admin/inventory-management/items/stock-inventory-model"));
const item_details_serial_number_1 = __importDefault(require("../../../../model/admin/inventory-management/items/item-details-serial-number"));
const job_order_1 = require("../../../../constants/admin/job-order");
const items_1 = require("../../../../constants/items");
const items_master_model_1 = __importDefault(require("../../../../model/admin/inventory-management/items/items-master-model"));
const controller = new base_controller_1.default();
class ItemDetailsController extends base_controller_1.default {
    async findAll(req, res) {
        const { itemMasterSlug = '', branchId = '', itemMasterId, stock = '0', getstockbranch = '0', itemType, page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: (0, lodash_1.isArray)(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { itemMasterTitle: keywordRegex },
                    { itemTitle: keywordRegex }
                ],
                ...query
            };
        }
        if (itemMasterId) {
            query.itemMasterId = new mongoose_1.default.Types.ObjectId(itemMasterId);
        }
        if (branchId) {
            query = {
                ...query, "stockInventory.branchId": new mongoose_1.default.Types.ObjectId(branchId)
            };
        }
        let branchStockQuery = { _id: { $exists: true } };
        if (stock === '1' && branchId) {
            branchStockQuery = {
                ...branchStockQuery, "branchId": new mongoose_1.default.Types.ObjectId(branchId)
            };
        }
        if (itemType) {
            query.itemType = itemType;
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const itemMaster = await item_details_service_1.default.findAllItem({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                branchStockQuery,
                getstockbranch,
                itemMasterSlug,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: itemMaster.items,
                totalCount: itemMaster.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching item master' });
        }
    }
    async ItemDetail(req, res) {
        try {
            const itemDetailId = req.params.id;
            if (!itemDetailId) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Item detail Id not provided! Please try again with an item detail Id.',
                });
            }
            const validatedData = await items_master_schema_1.itemDetailSchema.safeParseAsync({ ...req.body, _id: itemDetailId });
            if (validatedData.success) {
                const updatedItemDetailData = validatedData.data;
                const updatedItemDetail = await item_details_service_1.default.updateItem(itemDetailId, { ...updatedItemDetailData, updatedAt: new Date() });
                if (updatedItemDetail) {
                    // if (isArray(updatedItemDetailData.stockInventory) && size(updatedItemDetailData.stockInventory) > 0) {
                    //     for (const inventory of updatedItemDetailData.stockInventory) {
                    //         const { branchCode, itemStock } = inventory;
                    //         const branch = await BranchModel.findOne({ branchCode: branchCode });
                    //         if (!branch) {
                    //             if (updatedItemDetail) {
                    //                 await ItemsDetailModel.deleteOne({ _id: updatedItemDetail._id });
                    //             }
                    //             return controller.sendErrorResponse(res, 200, {
                    //                 message: `Branch code ${branchCode} does not exist.`,
                    //             });
                    //         }
                    //         const branchId = branch._id;
                    //         await StockInventoryModel.updateOne(
                    //             { itemId: updatedItemDetail._id, branchId },
                    //             {
                    //                 itemMasterId: updatedItemDetail.itemMasterId,
                    //                 itemId: updatedItemDetail._id,
                    //                 branchId,
                    //                 itemStock
                    //             },
                    //             { upsert: true }
                    //         );
                    //     }
                    // }
                    return controller.sendSuccessResponse(res, {
                        requestedData: updatedItemDetail,
                        message: 'Item detail updated successfully!',
                    }, 200, {
                        sourceFromId: updatedItemDetail._id,
                        sourceFrom: task_log_1.adminTaskLog.inventoryManagement.items.itemDetail,
                        activity: task_log_1.adminTaskLogActivity.update,
                        activityStatus: task_log_1.adminTaskLogStatus.success,
                        activityComment: 'Item detail updated successfully',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 404, {
                        message: 'Item detail Id not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors),
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating item detail',
            });
        }
    }
    async createItemDetail(req, res) {
        try {
            const productDetailData = await items_master_schema_1.itemDetailSchema.parseAsync(req.body);
            let successProducts = [];
            for (const productData of productDetailData) {
                const { itemMasterId, items, bomMasterQuantity, itemType = items_1.itemTypes.normal, itemMasterPrice, } = productData;
                const user = res.locals.user;
                if ((0, lodash_1.isArray)(items) && (0, lodash_1.size)(items) > 0) {
                    for (const item of items) {
                        const { warrantyId, itemTitle, itemCode, stockInventory, barCode, hsnCode, unit, itemPrice, itemTypeQuantity, measurements, itemDescription, itemImageUrl, qualityType, status, } = item;
                        let totalItemStock = 0;
                        if ((0, lodash_1.isArray)(stockInventory) && (0, lodash_1.size)(stockInventory) > 0) {
                            totalItemStock = stockInventory.reduce((sum, inventory) => sum + (inventory.itemStock || 0), 0);
                        }
                        const itemDetailData = {
                            itemMasterId,
                            warrantyId,
                            itemTitle,
                            slug: (0, helpers_1.slugifyManually)(itemTitle),
                            itemCode,
                            itemType,
                            itemTypeQuantity: itemTypeQuantity || 1,
                            itemPrice: itemPrice || 0,
                            totalItemStock,
                            barCode: barCode || '',
                            hsnCode: hsnCode || '',
                            unit: unit || '',
                            measurements: {
                                weight: measurements?.weight || '',
                                height: measurements?.height || '',
                                length: measurements?.length || '',
                                width: measurements?.width || '',
                            },
                            itemDescription: itemDescription || '',
                            itemImageUrl: itemImageUrl || '',
                            qualityType: qualityType || '',
                            status: status || '1',
                            createdBy: user._id,
                            updatedBy: user._id,
                            updatedAt: new Date(),
                        };
                        const bulkResult = await item_details_model_1.default.updateOne({ itemCode }, {
                            $set: itemDetailData,
                            $setOnInsert: { createdAt: new Date() },
                        }, { upsert: true });
                        let updatedItem;
                        if (bulkResult.upsertedId) {
                            updatedItem = await item_details_model_1.default.findOne({ _id: bulkResult.upsertedId });
                        }
                        else {
                            updatedItem = await item_details_model_1.default.findOne({ itemCode });
                        }
                        if (updatedItem) {
                            successProducts.push(updatedItem);
                        }
                    }
                }
                if (itemType === items_1.itemTypes.bom) {
                    await items_master_model_1.default.updateOne({ _id: new mongoose_1.default.Types.ObjectId(itemMasterId) }, {
                        $set: { bomMasterQuantity: bomMasterQuantity || 0, itemMasterPrice: itemMasterPrice || 0 },
                    });
                }
            }
            if ((0, lodash_1.isEmpty)(successProducts)) {
                return controller.sendErrorResponse(res, 500, {
                    message: 'Something went wrong while creating item detail',
                });
            }
            return controller.sendSuccessResponse(res, {
                requestedData: successProducts,
                message: 'Item detail created successfully!'
            });
        }
        catch (error) {
            if (error instanceof zod_1.ZodError) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(error.errors),
                });
            }
            if (error.errors && error.errors.itemTitle) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: { itemTitle: error.errors.itemTitle.properties.message },
                });
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while creating item detail',
            });
        }
    }
    async findOne(req, res) {
        try {
            const itemId = req.params.id;
            if (itemId) {
                const itemDetail = await item_details_service_1.default.findOneItem(itemId);
                if (itemDetail) {
                    return controller.sendSuccessResponse(res, {
                        requestedData: itemDetail,
                        message: 'Success',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 404, {
                        message: 'Item detail not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Item detail ID not provided!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async updateItemStock(req, res) {
        const validatedData = await items_master_schema_1.itemStockUpdateSchema.safeParseAsync(req.body);
        if (validatedData.success) {
            const { stockInventory, } = validatedData.data;
            if ((0, lodash_1.isArray)(stockInventory) && (0, lodash_1.size)(stockInventory) > 0) {
                const branchCodes = stockInventory.map(inv => inv.branchCode);
                const transferBranchCodes = stockInventory.flatMap(inv => inv.transferBranches ? inv.transferBranches.map(tb => tb.transferBranchCode) : []);
                const itemCodes = stockInventory.map(inv => inv.itemCode);
                const allBranchCodes = [...new Set([...branchCodes, ...transferBranchCodes])];
                const branches = await branch_model_1.default.find({ branchCode: { $in: allBranchCodes } }).lean();
                const itemsDetails = await item_details_model_1.default.find({ itemCode: { $in: itemCodes } }).lean();
                const branchMap = new Map(branches.map(branch => [branch.branchCode, branch]));
                const itemMap = new Map(itemsDetails.map(item => [item.itemCode, item]));
                const stockInventoryBulkOperations = [];
                const serialAndBatchBulkInsert = [];
                const errorMessages = [];
                const addToBulkInsert = (itemsDetail, branch, stockType, serialNumbers, batchCodes, itemStock) => {
                    if (serialNumbers && (0, lodash_1.size)(serialNumbers) > 0) {
                        serialNumbers.forEach((serialNumber) => {
                            serialAndBatchBulkInsert.push({
                                itemMasterId: itemsDetail.itemMasterId,
                                itemId: itemsDetail._id,
                                branchId: branch._id,
                                serialNumberType: job_order_1.serialNumberTypes.serialNumber,
                                serialNumber,
                                stockType,
                                stockCount: 1,
                            });
                        });
                    }
                    else if (batchCodes && (0, lodash_1.size)(batchCodes) > 0) {
                        batchCodes.forEach((batchCode) => {
                            serialAndBatchBulkInsert.push({
                                itemMasterId: itemsDetail.itemMasterId,
                                itemId: itemsDetail._id,
                                branchId: branch._id,
                                serialNumberType: job_order_1.serialNumberTypes.batch,
                                batchNumber: batchCode.batchCode,
                                stockType,
                                stockCount: batchCode.itemStock,
                            });
                        });
                    }
                };
                for (const inventory of stockInventory) {
                    const { itemCode, branchCode, itemStock, serialNumbers, batchCodes, transferBranches, usedItems } = inventory;
                    const branch = branchMap.get(branchCode);
                    const itemsDetail = itemMap.get(itemCode);
                    if (!branch) {
                        errorMessages.push({ branchCode: `Branch code ${branchCode} does not exist.` });
                        continue;
                    }
                    if (!itemsDetail) {
                        errorMessages.push({ itemCode: `Item code ${itemCode} does not exist.` });
                        continue;
                    }
                    let stockUpdate = {
                        $set: {
                            itemMasterId: itemsDetail.itemMasterId,
                            itemId: itemsDetail._id,
                            branchId: branch._id,
                        },
                    };
                    if (serialNumbers && (0, lodash_1.size)(serialNumbers) > 0) {
                        stockUpdate.$inc = { itemStock: (0, lodash_1.size)(serialNumbers) };
                    }
                    else if (batchCodes && (0, lodash_1.size)(batchCodes) > 0) {
                        const totalItemStock = batchCodes.reduce((sum, batch) => sum + batch.itemStock, 0);
                        stockUpdate.$inc = { itemStock: totalItemStock };
                    }
                    else if (transferBranches && (0, lodash_1.size)(transferBranches) > 0) {
                        for (const transferBranch of transferBranches) {
                            const { transferBranchCode, serialNumber, batchCode, itemStock } = transferBranch;
                            if (transferBranchCode) {
                                const checkBranch = branchMap.get(transferBranchCode);
                                if (!checkBranch) {
                                    errorMessages.push({ branchCode: `Branch code ${transferBranchCode} does not exist.` });
                                    continue;
                                }
                                if (!batchCode && serialNumber) {
                                    await item_details_serial_number_1.default.updateOne({ branchId: checkBranch._id, serialNumber }, { $inc: { stockOutCount: itemStock }, $set: { stockType: job_order_1.stockTypes.out } });
                                    await item_details_serial_number_1.default.updateOne({ branchId: checkBranch._id, serialNumber }, { $inc: { stockCount: itemStock }, $set: { stockType: job_order_1.stockTypes.in }, }, { upsert: true });
                                }
                                else if (!serialNumber && batchCode) {
                                    await item_details_serial_number_1.default.updateOne({ branchId: checkBranch._id, batchNumber: batchCode }, { $inc: { stockCount: -itemStock, stockOutCount: itemStock } });
                                    await item_details_serial_number_1.default.updateOne({ branchId: checkBranch._id, batchNumber: batchCode }, { $inc: { stockCount: itemStock } }, { upsert: true });
                                }
                                else if (!serialNumber && !batchCode) {
                                    await item_details_serial_number_1.default.updateOne({ branchId: checkBranch._id, }, { $inc: { stockCount: -itemStock, stockOutCount: itemStock } });
                                    await item_details_serial_number_1.default.updateOne({ branchId: checkBranch._id }, { $inc: { stockCount: itemStock } }, { upsert: true });
                                }
                            }
                        }
                    }
                    else if (usedItems && (0, lodash_1.size)(usedItems) > 0) {
                        // const totalUsedItemStock = usedItems.reduce((sum, usedItem) => sum - usedItem.itemStock, 0);
                        // stockUpdate.$inc = { itemStock: totalUsedItemStock };
                        let totalUsedItemStock = 0;
                        for (const usedItem of usedItems) {
                            const { serialNumber, batchCode, itemStock } = usedItem;
                            if (!batchCode && serialNumber) {
                                const item = await item_details_serial_number_1.default.findOne({ branchId: branch._id, serialNumber });
                                if (item) {
                                    const newStockCount = Math.max(item.stockCount - itemStock, 0);
                                    await item_details_serial_number_1.default.updateOne({ _id: item._id }, {
                                        $set: { stockCount: newStockCount, stockType: job_order_1.stockTypes.out },
                                        $inc: { stockOutCount: itemStock },
                                    });
                                    totalUsedItemStock -= itemStock;
                                }
                            }
                            else if (!serialNumber && batchCode) {
                                const item = await item_details_serial_number_1.default.findOne({ branchId: branch._id, batchNumber: batchCode });
                                if (item) {
                                    const newStockCount = Math.max(item.stockCount - itemStock, 0);
                                    await item_details_serial_number_1.default.updateOne({ _id: item._id }, {
                                        $set: { stockCount: newStockCount },
                                        $inc: { stockOutCount: itemStock }
                                    });
                                    totalUsedItemStock -= itemStock;
                                }
                            }
                            else if (!serialNumber && !batchCode) {
                                const item = await item_details_serial_number_1.default.findOne({ branchId: branch._id });
                                if (item) {
                                    if (!item.serialNumber && !item.batchNumber) {
                                        const newStockCount = Math.max(item.stockCount - itemStock, 0);
                                        await item_details_serial_number_1.default.updateOne({ _id: item._id }, {
                                            $set: { stockCount: newStockCount },
                                            $inc: { stockOutCount: itemStock },
                                        });
                                        totalUsedItemStock -= itemStock;
                                    }
                                }
                            }
                        }
                        if (totalUsedItemStock !== 0) {
                            stockUpdate.$set.itemStock = totalUsedItemStock;
                        }
                    }
                    else {
                        stockUpdate.$set.itemStock = itemStock;
                    }
                    stockInventoryBulkOperations.push({
                        updateOne: {
                            filter: { itemId: itemsDetail._id, branchId: branch._id },
                            update: stockUpdate,
                            upsert: true,
                        },
                    });
                    addToBulkInsert(itemsDetail, branch, job_order_1.stockTypes.in, serialNumbers, batchCodes, itemStock);
                }
                if ((0, lodash_1.size)(stockInventoryBulkOperations) > 0) {
                    await stock_inventory_model_1.default.bulkWrite(stockInventoryBulkOperations);
                }
                if ((0, lodash_1.size)(serialAndBatchBulkInsert) > 0) {
                    await item_details_serial_number_1.default.insertMany(serialAndBatchBulkInsert);
                }
                const stockAggregation = await stock_inventory_model_1.default.aggregate([
                    {
                        $group: {
                            _id: "$itemId",
                            totalItemStock: { $sum: "$itemStock" },
                        },
                    },
                ]);
                const bulkUpdateOperations = stockAggregation.map(stock => ({
                    updateOne: {
                        filter: { _id: stock._id },
                        update: { $set: { totalItemStock: stock.totalItemStock } }
                    }
                }));
                if (bulkUpdateOperations.length > 0) {
                    await item_details_model_1.default.bulkWrite(bulkUpdateOperations);
                }
                if ((0, lodash_1.size)(errorMessages) > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: errorMessages,
                    });
                }
                else {
                    return controller.sendSuccessResponse(res, {
                        message: 'Item detail stock updated successfully!',
                    }, 200);
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Stock inventory is required!',
                });
            }
        }
        else {
            return controller.sendErrorResponse(res, 200, {
                message: 'Validation error',
                validation: (0, helpers_1.formatZodError)(validatedData.error.errors),
            });
        }
    }
    async statusChange(req, res) {
        try {
            const validatedData = await items_master_schema_1.itemDetailStatusSchema.safeParseAsync(req.body);
            if (validatedData.success) {
                const itemDetailId = req.params.id;
                if (itemDetailId) {
                    const { status } = req.body;
                    const updatedItemDetailData = { status };
                    const updatedItemDetail = await item_details_service_1.default.updateItem(itemDetailId, updatedItemDetailData);
                    if (updatedItemDetail) {
                        return controller.sendSuccessResponse(res, {
                            requestedData: updatedItemDetail,
                            message: 'Item detail status updated successfully!'
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 404, {
                            message: 'Item detail ID not found!',
                        });
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Item detail ID not provided! Please try again with an item detail ID.',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating item detail status'
            });
        }
    }
    async destroy(req, res) {
        try {
            const itemDetailId = req.params.id;
            if (itemDetailId) {
                const itemDetail = await item_details_service_1.default.findOneItem(itemDetailId);
                if (itemDetail) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cannot delete this item detail.',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This item detail not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Item detail ID not provided!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while trying to delete the item detail.'
            });
        }
    }
}
exports.default = new ItemDetailsController();
