"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const zod_1 = require("zod");
const lodash_1 = require("lodash");
const helpers_1 = require("../../../../utils/helpers");
const task_log_1 = require("../../../../constants/admin/task-log");
const items_master_schema_1 = require("../../../../utils/schemas/admin/inventory-management/items/items-master-schema");
const items_master_service_1 = __importDefault(require("../../../../services/admin/inventory-management/items/items-master-service"));
const base_controller_1 = __importDefault(require("../../base-controller"));
const category_model_1 = __importDefault(require("../../../../model/admin/inventory-management/category-model"));
const category_service_1 = __importDefault(require("../../../../services/admin/inventory-management/category-service"));
const items_master_model_1 = __importDefault(require("../../../../model/admin/inventory-management/items/items-master-model"));
const items_1 = require("../../../../constants/items");
const item_details_model_1 = __importDefault(require("../../../../model/admin/inventory-management/items/item-details-model"));
const controller = new base_controller_1.default();
class ItemsMasterController extends base_controller_1.default {
    async findAll(req, res) {
        const { categoryId = '', page_size = 1, itemType, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (keyword) {
            const escapedKeyword = keyword.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
            const keywordRegex = new RegExp(`\\b${escapedKeyword}`, 'i');
            // const  keywordRegex = new RegExp(`^${keyword}`, 'i');
            query = {
                $or: [
                    { itemMasterTitle: { $regex: keywordRegex } },
                    { itemMasterCode: keyword },
                    { itemType: keyword },
                    { itemMasterDescription: { $regex: keywordRegex } },
                    { slug: { $regex: keywordRegex } },
                ],
                ...query
            };
        }
        if (categoryId) {
            query.categoryId = new mongoose_1.default.Types.ObjectId(categoryId);
        }
        if (itemType) {
            query.itemType = itemType;
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const itemMaster = await items_master_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: itemMaster.data,
                totalCount: itemMaster.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching item master' });
        }
    }
    async create(req, res) {
        try {
            const productMasterData = await items_master_schema_1.itemMasterSchema.parseAsync(req.body);
            let successsProducts = [];
            const categoryIds = productMasterData.map((product) => product.categoryId);
            const categoriesMap = await category_model_1.default.find({ _id: { $in: categoryIds } })
                .select('_id')
                .lean()
                .then(categories => {
                return categories.reduce((acc, category) => {
                    acc[category._id.toString()] = category;
                    return acc;
                }, {});
            });
            const missingCategories = categoryIds.filter((categoryId) => !categoriesMap[categoryId]);
            if (missingCategories.length > 0) {
                return controller.sendErrorResponse(res, 200, {
                    message: `The following categories are not found: ${missingCategories.join(', ')}`,
                });
            }
            for (const productData of productMasterData) {
                const { categoryId, itemMasterTitle, itemMasterCode, itemType = items_1.itemTypes.normal, itemMasterDescription, itemMasterImageUrl, itemMasterAttributes, status, } = productData;
                const user = res.locals.user;
                const categoryDetails = categoriesMap[categoryId];
                const itemMasterData = {
                    categoryId: categoryDetails._id,
                    itemMasterTitle: itemMasterTitle,
                    slug: (0, helpers_1.slugifyManually)(itemMasterTitle),
                    itemMasterCode,
                    itemType,
                    itemMasterDescription: itemMasterDescription || '',
                    itemMasterImageUrl: itemMasterImageUrl || '',
                    status: status || '1',
                    createdBy: user._id,
                    createdAt: new Date(),
                    updatedBy: user._id,
                    updatedAt: new Date(),
                };
                const newItemMaster = await items_master_model_1.default.create(itemMasterData);
                successsProducts.push(newItemMaster);
                for (let j = 0; j < itemMasterAttributes.length; j++) {
                    const attributeData = {
                        itemMasterId: newItemMaster._id,
                        attributeId: itemMasterAttributes[j].attributeId,
                        attributeDetailId: itemMasterAttributes[j].attributeDetailId
                    };
                    await items_master_service_1.default.createItemMasterAttributes(attributeData);
                }
            }
            return controller.sendSuccessResponse(res, {
                requestedData: successsProducts,
                message: 'Item master created successfully!'
            });
        }
        catch (error) {
            if (error instanceof zod_1.ZodError) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(error.errors),
                });
            }
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    const errorDetail = errors[key];
                    validationErrors[key] = {
                        message: errorDetail.message,
                        value: errorDetail.value,
                    };
                });
                if (Object.keys(validationErrors).length > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors,
                    });
                }
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'An error occurred while creating the item master.',
            });
        }
    }
    async updateBomItemQuanity(req, res) {
        try {
            const itemMasterId = req.params.id;
            if (!itemMasterId) {
                return controller.sendErrorResponse(res, 200, {
                    message: `Item master id is required`,
                });
            }
            const productMasterData = await items_master_schema_1.updateBomItemQuantitySchema.parseAsync(req.body);
            const itemMasterCodes = Array.from(new Set(productMasterData.map(item => item.itemMasterCode)));
            const existingItems = await items_master_model_1.default.find({ itemMasterCode: { $in: itemMasterCodes } }).select('_id itemType itemMasterCode bomMasterQuantity bomMasterOldQuantity itemMasterPrice').lean();
            const existingCodesSet = new Set(existingItems.map(item => item.itemMasterCode));
            const missingCodes = itemMasterCodes.filter(code => !existingCodesSet.has(code));
            if (missingCodes.length > 0) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Some itemMasterCodes do not exist in the itemsMaster collection.',
                    missingItemMasterCodes: missingCodes,
                });
            }
            const updatedItemMasterCodes = [];
            for (let updateData of productMasterData) {
                const { itemMasterCode, bomMasterQuantity, itemMasterPrice } = updateData;
                const itemMasterDetail = existingItems.find((item) => item.itemMasterCode === itemMasterCode);
                if (!(0, lodash_1.isEmpty)(itemMasterDetail)) {
                    const updatedItemMasterData = {};
                    if (bomMasterQuantity && bomMasterQuantity > 0) {
                        updatedItemMasterData.bomMasterQuantity = bomMasterQuantity;
                        updatedItemMasterData.bomMasterOldQuantity = itemMasterDetail.bomMasterQuantity;
                    }
                    if (itemMasterPrice && itemMasterPrice > 0) {
                        updatedItemMasterData.itemMasterPrice = itemMasterPrice;
                    }
                    const updatedJobDetailData = await items_master_model_1.default.findOneAndUpdate({ _id: itemMasterDetail._id }, { $set: updatedItemMasterData });
                    updatedItemMasterCodes.push(itemMasterCode);
                    if (!(0, lodash_1.isEmpty)(updatedJobDetailData) && bomMasterQuantity && bomMasterQuantity > 0) {
                        const calculatedBomChangedPercentage = ((bomMasterQuantity - itemMasterDetail.bomMasterQuantity) / itemMasterDetail.bomMasterQuantity) * 100;
                        if (calculatedBomChangedPercentage !== 0) {
                            const itemDetails = await item_details_model_1.default.find({ itemMasterId: updatedJobDetailData._id });
                            if ((0, lodash_1.isArray)(itemDetails) && (0, lodash_1.size)(itemDetails) > 0) {
                                for (let itemDetail of itemDetails) {
                                    const newQuantity = itemDetail.itemTypeQuantity + (itemDetail.itemTypeQuantity * calculatedBomChangedPercentage / 100);
                                    const roundedQuantity = Math.max(0.999, Math.round(newQuantity * 1000) / 1000);
                                    console.log(itemDetail.itemTypeQuantity, 'itemTypeQuantity', roundedQuantity, newQuantity);
                                    await item_details_model_1.default.updateOne({ _id: itemDetail._id }, { $set: { itemTypeQuantity: roundedQuantity } });
                                }
                            }
                        }
                    }
                }
            }
            return controller.sendSuccessResponse(res, {
                requestedData: updatedItemMasterCodes,
                message: 'Item master updation successfully completed!'
            });
        }
        catch (error) {
            if (error instanceof zod_1.ZodError) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(error.errors),
                });
            }
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    const errorDetail = errors[key];
                    validationErrors[key] = {
                        message: errorDetail.message,
                        value: errorDetail.value,
                    };
                });
                if (Object.keys(validationErrors).length > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors,
                    });
                }
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'An error occurred while creating the item master.',
            });
        }
    }
    async findOne(req, res) {
        try {
            const itemMasterId = req.params.id;
            if (itemMasterId) {
                const itemMaster = await items_master_service_1.default.findOne(itemMasterId);
                return controller.sendSuccessResponse(res, {
                    requestedData: itemMaster,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Item master Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async findOneItemMasterDetails(req, res) {
        try {
            const itemMasterId = req.params.id;
            if (itemMasterId) {
                let itemMaster = await items_master_service_1.default.findOne(itemMasterId);
                if ((0, lodash_1.isEmpty)(itemMaster)) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Item master details not found!',
                    });
                }
                if (!(0, lodash_1.isEmpty)(itemMaster.category)) {
                    let query = { _id: { $exists: true } };
                    query._id = itemMaster.category._id;
                    const categories = await category_service_1.default.findAllParentCategories({ query });
                    itemMaster = {
                        ...itemMaster,
                        category: categories
                    };
                }
                return controller.sendSuccessResponse(res, {
                    requestedData: itemMaster,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Item master Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const itemMasterId = req.params.id;
            if (!itemMasterId) {
                return controller.sendErrorResponse(res, 400, {
                    message: 'Item master Id not provided! Please try again with an item master Id.',
                });
            }
            const validatedData = await items_master_schema_1.itemMasterSchema.safeParseAsync({ ...req.body, _id: itemMasterId });
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 400, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors),
                });
            }
            const updatedItemMasterData = validatedData.data;
            const updatedItemMaster = await items_master_service_1.default.update(itemMasterId, { ...updatedItemMasterData, updatedAt: new Date() });
            if (updatedItemMaster) {
                return controller.sendSuccessResponse(res, {
                    requestedData: updatedItemMaster,
                    message: 'Item master updated successfully!',
                }, 200, {
                    sourceFromId: updatedItemMaster._id,
                    sourceFrom: task_log_1.adminTaskLog.inventoryManagement.items.itemMaster,
                    activity: task_log_1.adminTaskLogActivity.update,
                    activityStatus: task_log_1.adminTaskLogStatus.success,
                    activityComment: 'Item master updated successfully',
                });
            }
            else {
                return controller.sendErrorResponse(res, 404, {
                    message: 'Item master Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating item master',
            });
        }
    }
    async statusChange(req, res) {
        try {
            const validatedData = items_master_schema_1.itemMasterStatusSchema.safeParse(req.body);
            if (validatedData.success) {
                const itemMasterId = req.params.id;
                if (itemMasterId) {
                    let { status } = req.body;
                    const updateditemMasterData = { status };
                    const updateditemMaster = await items_master_service_1.default.update(itemMasterId, updateditemMasterData);
                    if (updateditemMaster) {
                        return controller.sendSuccessResponse(res, {
                            requestedData: updateditemMaster,
                            message: 'Item master status updated successfully!'
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'itemMaster Id not found!',
                        }, req);
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Item master Id not found! Please try again with Item master id',
                    }, req);
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                }, req);
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating Item master'
            }, req);
        }
    }
    async destroy(req, res) {
        try {
            const itemMasterId = req.params.id;
            if (itemMasterId) {
                const itemMaster = await items_master_service_1.default.findOne(itemMasterId);
                if (itemMaster) {
                    // await ItemsMasterInventoryService.destroy(itemMasterId);
                    // return controller.sendSuccessResponse(res, { message: 'ItemMaster deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this Item master',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This item master details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Item master id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting item master' });
        }
    }
}
exports.default = new ItemsMasterController();
