"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const base_controller_1 = __importDefault(require("../../base-controller"));
const job_order_service_1 = __importDefault(require("../../../../services/admin/job-order/job-order-service"));
const items_master_schema_1 = require("../../../../utils/schemas/admin/inventory-management/items/items-master-schema");
const helpers_1 = require("../../../../utils/helpers");
const spare_requests_and_transfers_model_1 = __importDefault(require("../../../../model/admin/inventory-management/items/spare-requests-and-transfers-model"));
const lodash_1 = require("lodash");
const controller = new base_controller_1.default();
class ItemsOtherController extends base_controller_1.default {
    async findAllRequestedItems(req, res) {
        try {
            const { jobOrderId = '', jobOrderDetailId = '', branchId = '', page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
            let query = { _id: { $exists: true } };
            if (jobOrderId) {
                query.jobMasterId = new mongoose_1.default.Types.ObjectId(jobOrderId);
            }
            if (branchId) {
                query.branchId = new mongoose_1.default.Types.ObjectId(branchId);
            }
            const sort = {};
            if (sortby && sortorder) {
                sort[sortby] = sortorder === 'desc' ? -1 : 1;
            }
            const [jobOrderDetails] = await job_order_service_1.default.findAllRequestedItems({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: jobOrderDetails.data,
                totalCount: jobOrderDetails.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async itemRequestAccept(req, res) {
        try {
            const validatedData = items_master_schema_1.itemRequestAcceptFormSchema.safeParse(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
            const { requestId, requestedBranchId, batchSerial, serialNumbers, acceptedQuantity, requestedStatus, requestedQuantity } = validatedData.data;
            const requestedDetail = await spare_requests_and_transfers_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(requestId) });
            if ((0, lodash_1.isEmpty)(requestedDetail)) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Requested details not found!',
                });
            }
            const sumStockCount = (0, lodash_1.isArray)(batchSerial) && batchSerial.length > 0 ? batchSerial.reduce((acc, current) => acc + (current.quantity || 0), 0) : 0;
            const updaterequestedData = {
                transferredQuantity: (0, lodash_1.size)(serialNumbers) > 0 ? (0, lodash_1.size)(serialNumbers) : (sumStockCount > 0 ? sumStockCount : (0, lodash_1.isNumber)(acceptedQuantity)),
                requestedStatus: requestedStatus,
                requestedStatusAt: new Date()
            };
            const retVal = await spare_requests_and_transfers_model_1.default.findByIdAndUpdate(new mongoose_1.default.Types.ObjectId(requestId), updaterequestedData, { new: true, useFindAndModify: false });
            return controller.sendSuccessResponse(res, {
                requestedData: retVal,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    validationErrors[key] = errors[key].message;
                });
                if (Object.keys(validationErrors).length > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors
                    });
                }
            }
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
}
exports.default = new ItemsOtherController();
