"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const zod_1 = require("zod");
const lodash_1 = require("lodash");
const helpers_1 = require("../../../../utils/helpers");
const items_master_schema_1 = require("../../../../utils/schemas/admin/inventory-management/items/items-master-schema");
const products_service_1 = __importDefault(require("../../../../services/admin/inventory-management/products/products.service"));
const base_controller_1 = __importDefault(require("../../base-controller"));
const category_model_1 = __importDefault(require("../../../../model/admin/inventory-management/category-model"));
const category_service_1 = __importDefault(require("../../../../services/admin/inventory-management/category-service"));
const items_master_model_1 = __importDefault(require("../../../../model/admin/inventory-management/items/items-master-model"));
const items_1 = require("../../../../constants/items");
const item_details_model_1 = __importDefault(require("../../../../model/admin/inventory-management/items/item-details-model"));
const product_schema_1 = require("../../../../utils/schemas/admin/inventory-management/products/product.schema");
const job_order_1 = require("../../../../constants/admin/job-order");
const branch_model_1 = __importDefault(require("../../../../model/admin/operations/branch-model"));
const products_serial_number_1 = __importDefault(require("../../../../model/admin/inventory-management/products/products-serial-number"));
const products_model_1 = __importDefault(require("../../../../model/admin/inventory-management/products/products.model"));
const products_stock_inventory_model_1 = __importDefault(require("../../../../model/admin/inventory-management/products/products-stock-inventory-model"));
const category_config_1 = require("../../../../utils/config/admin/inventory-management/category-config");
const category_1 = require("../../../../utils/admin/category");
class ProductsController extends base_controller_1.default {
    constructor(productsModel = products_model_1.default, inventoryService, branchModel = branch_model_1.default, productsSerialNumberModel = products_serial_number_1.default, productStockInventoryModel = products_stock_inventory_model_1.default) {
        super();
        this.productsModel = productsModel;
        this.inventoryService = inventoryService || new products_service_1.default();
        this.branchModel = branchModel;
        this.productsSerialNumberModel = productsSerialNumberModel;
        this.productStockInventoryModel = productStockInventoryModel;
    }
    async findAll(req, res) {
        const { categoryId = '', isNotFetchCategory = '', page_size = 1, itemType, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        query.status = status && status !== '' ? { $in: Array.isArray(status) ? status : [status] } : '1';
        if (keyword) {
            const escapedKeyword = keyword.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
            const keywordRegex = new RegExp(`\\b${escapedKeyword}`, 'i');
            query = {
                $or: [
                    { itemTitle: { $regex: keywordRegex } },
                    { itemMasterCode: keyword },
                    { itemType: keyword },
                    { itemMasterDescription: { $regex: keywordRegex } },
                    { slug: { $regex: keywordRegex } },
                ],
                ...query
            };
        }
        if (categoryId)
            query.categoryId = new mongoose_1.default.Types.ObjectId(categoryId);
        if (itemType)
            query.itemType = itemType;
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const itemMaster = await this.inventoryService.findAll({
                page: parseInt(page_size, 10),
                limit: parseInt(limit, 10),
                query,
                sort,
                isNotFetchCategory
            });
            return this.sendSuccessResponse(res, {
                requestedData: itemMaster.data,
                totalCount: itemMaster.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return this.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while fetching item master'
            });
        }
    }
    async getFilterData(req, res) {
        try {
            const { categoryId = '', page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
            let query = { _id: { $exists: true } };
            const user = await res.locals.user;
            if (keyword) {
                const keywordRegex = new RegExp(`^${keyword}`, 'i');
                query.$or.push({ 'itemCode': Number(keyword) }, { 'itemType': { $regex: keywordRegex } }, { 'itemTitle': { $regex: keywordRegex } }, { 'itemAttributes.attributeName': { $regex: keywordRegex } }, { 'itemAttributes.itemValue': { $regex: keywordRegex } });
            }
            if (categoryId) {
                let categoryBatchIds = [];
                categoryBatchIds.push(new mongoose_1.default.Types.ObjectId(categoryId));
                const uniqueCategoryIds = [...new Set(categoryBatchIds)];
                var categoriesWithChilleds = await category_model_1.default.aggregate((0, category_config_1.categoriesWithChildrenInArrayAggregation)(uniqueCategoryIds));
                const categoryIds = (0, category_1.extractAllCategoryIds)(categoriesWithChilleds);
                query = { ...query, categoryId: { $in: categoryIds } };
            }
            const sort = {};
            if (sortby && sortorder) {
                sort[sortby] = sortorder === 'desc' ? -1 : 1;
            }
            const filterData = await this.inventoryService.findAllFilterData({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return this.sendSuccessResponse(res, {
                requestedData: filterData.data,
                totalCount: filterData.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return this.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async getStockInventory(req, res) {
        const { page_size = 1, limit = 10, sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (keyword) {
            const escapedKeyword = keyword.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
            const keywordRegex = new RegExp(`\\b${escapedKeyword}`, 'i');
            query = {
                $or: [
                    { itemCode: { $regex: keywordRegex } },
                    { branchCode: keyword },
                ],
                ...query
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const stockInventory = await this.inventoryService.getStockInventory({
                page: parseInt(page_size, 10),
                limit: parseInt(limit, 10),
                query,
                sort
            });
            return this.sendSuccessResponse(res, {
                requestedData: stockInventory.data,
                totalCount: stockInventory.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return this.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while fetching item master'
            });
        }
    }
    async create(req, res) {
        try {
            const productMasterData = await product_schema_1.productSchema.parseAsync(req.body);
            let successsProducts = [];
            const categoryIds = productMasterData.map((product) => product.categoryId);
            const categoriesMap = await category_model_1.default.find({ _id: { $in: categoryIds } }).select('_id').lean().then(categories => {
                return categories.reduce((acc, category) => {
                    acc[category._id.toString()] = category;
                    return acc;
                }, {});
            });
            const missingCategories = categoryIds.filter((categoryId) => !categoriesMap[categoryId]);
            if (missingCategories.length > 0) {
                return this.sendErrorResponse(res, 200, { message: `The following categories are not found: ${missingCategories.join(', ')}` });
            }
            for (const productData of productMasterData) {
                const { categoryId, itemTitle, itemType = items_1.itemTypes.normal, itemAttributes, ...restValues } = productData;
                const user = res.locals.user;
                const categoryDetails = categoriesMap[categoryId];
                const itemMasterData = {
                    categoryId: categoryDetails._id,
                    itemTitle: itemTitle,
                    slug: (0, helpers_1.slugifyManually)(itemTitle),
                    itemType,
                    itemAttributes,
                    ...restValues,
                    createdBy: user._id,
                    createdAt: new Date(),
                    updatedBy: user._id,
                    updatedAt: new Date(),
                };
                const newItemMaster = await this.inventoryService.manageProduct(itemMasterData);
                successsProducts.push(newItemMaster);
                await this.inventoryService.deleteProductAttribute(newItemMaster._id);
                for (let j = 0; j < itemAttributes.length; j++) {
                    const attributeData = {
                        productId: newItemMaster._id,
                        itemCode: newItemMaster.itemCode,
                        attributeId: itemAttributes[j].attributeId,
                        attributeDetailId: itemAttributes[j].attributeDetailId,
                        attributeName: itemAttributes[j].attributeName,
                        itemValue: itemAttributes[j].itemValue,
                    };
                    await this.inventoryService.createProductAttributes(attributeData);
                }
            }
            return this.sendSuccessResponse(res, {
                requestedData: successsProducts,
                message: 'Item master created successfully!',
            });
        }
        catch (error) {
            if (error instanceof zod_1.ZodError) {
                return this.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(error.errors),
                });
            }
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    const errorDetail = errors[key];
                    validationErrors[key] = {
                        message: errorDetail.message,
                        value: errorDetail.value,
                    };
                });
                if (Object.keys(validationErrors).length > 0) {
                    return this.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors,
                    });
                }
            }
            return this.sendErrorResponse(res, 500, {
                message: error.message || 'An error occurred while creating the item master.',
            });
        }
    }
    async updateProductStock(req, res) {
        const validatedData = await product_schema_1.productStockUpdateSchema.safeParseAsync(req.body);
        if (validatedData.success) {
            const { stockInventory } = validatedData.data;
            if ((0, lodash_1.isArray)(stockInventory) && (0, lodash_1.size)(stockInventory) > 0) {
                const branchCodes = stockInventory.map(inv => inv.branchCode);
                const transferBranchCodes = stockInventory.flatMap(inv => inv.transferBranches ? inv.transferBranches.map(tb => tb.transferBranchCode) : []);
                const itemCodes = stockInventory.map(inv => inv.itemCode);
                const allBranchCodes = [...new Set([...branchCodes, ...transferBranchCodes])];
                const branches = await this.branchModel.find({ branchCode: { $in: allBranchCodes } }).lean();
                const productDetails = await this.productsModel.find({ itemCode: { $in: itemCodes } }).select('_id itemCode itemType').lean();
                const branchMap = new Map(branches.map(branch => [branch.branchCode, branch]));
                const itemMap = new Map(productDetails.map(item => [item.itemCode, item]));
                const stockInventoryBulkOperations = [];
                const serialAndBatchBulkInsert = [];
                const errorMessages = [];
                const addToBulkInsert = (itemsDetail, branch, stockType, serialNumbers, batchCodes, itemStock) => {
                    if (serialNumbers && (0, lodash_1.size)(serialNumbers) > 0) {
                        serialNumbers.forEach((serialNumber) => {
                            serialAndBatchBulkInsert.push({
                                productId: itemsDetail._id,
                                branchId: branch._id,
                                serialNumberType: job_order_1.serialNumberTypes.serialNumber,
                                serialNumber,
                                stockType,
                                stockCount: 1,
                                branchCode: branch.branchCode,
                                itemCode: itemsDetail.itemCode
                            });
                        });
                    }
                    else if (batchCodes && (0, lodash_1.size)(batchCodes) > 0) {
                        batchCodes.forEach((batchCode) => {
                            serialAndBatchBulkInsert.push({
                                productId: itemsDetail._id,
                                branchId: branch._id,
                                serialNumberType: job_order_1.serialNumberTypes.batch,
                                batchNumber: batchCode.batchCode,
                                stockType,
                                stockCount: batchCode.itemStock,
                                branchCode: branch.branchCode,
                                itemCode: itemsDetail.itemCode
                            });
                        });
                    }
                };
                const productBulkOps = [];
                for (const inventory of stockInventory) {
                    const { itemCode, branchCode, itemStock, serialNumbers, batchCodes, transferBranches, usedItems } = inventory;
                    const branch = branchMap.get(branchCode);
                    const itemsDetail = itemMap.get(itemCode);
                    if (!branch) {
                        errorMessages.push({ branchCode: `Branch code ${branchCode} does not exist.` });
                        continue;
                    }
                    if (!itemsDetail) {
                        errorMessages.push({ itemCode: `Item code ${itemCode} does not exist.` });
                        continue;
                    }
                    let stockUpdate = {
                        $set: {
                            productId: itemsDetail._id,
                            branchId: branch._id,
                            branchCode: branch.branchCode,
                            itemCode: itemsDetail.itemCode,
                        },
                    };
                    let totalProductStock = 0;
                    if (serialNumbers && (0, lodash_1.size)(serialNumbers) > 0) {
                        stockUpdate.$inc = { itemStock: (0, lodash_1.size)(serialNumbers) };
                        totalProductStock = (0, lodash_1.size)(serialNumbers);
                    }
                    else if (batchCodes && (0, lodash_1.size)(batchCodes) > 0) {
                        const totalItemStock = batchCodes.reduce((sum, batch) => sum + batch.itemStock, 0);
                        stockUpdate.$inc = { itemStock: totalItemStock };
                        totalProductStock = totalItemStock;
                    }
                    else if (transferBranches && (0, lodash_1.size)(transferBranches) > 0) {
                        const bulkOps = [];
                        for (const transferBranch of transferBranches) {
                            const { transferBranchCode, serialNumber, batchCode, itemStock } = transferBranch;
                            if (transferBranchCode) {
                                const checkBranch = branchMap.get(transferBranchCode);
                                if (!checkBranch) {
                                    errorMessages.push({ branchCode: `Branch code ${transferBranchCode} does not exist.` });
                                    continue;
                                }
                                if (!batchCode && serialNumber) {
                                    bulkOps.push({ updateOne: { filter: { branchId: branch._id, serialNumber }, update: { $inc: { stockOutCount: itemStock }, $set: { stockType: job_order_1.stockTypes.out } }, } });
                                    bulkOps.push({ updateOne: { filter: { branchId: checkBranch._id, serialNumber }, update: { $inc: { stockCount: itemStock }, $set: { stockType: job_order_1.stockTypes.in, branchCode: checkBranch.branchCode, itemCode: itemsDetail.itemCode } }, upsert: true } });
                                }
                                else if (!serialNumber && batchCode) {
                                    bulkOps.push({ updateOne: { filter: { branchId: branch._id, batchNumber: batchCode }, update: { $inc: { stockCount: -itemStock, stockOutCount: itemStock } }, } });
                                    bulkOps.push({ updateOne: { filter: { branchId: checkBranch._id, batchNumber: batchCode }, update: { $inc: { stockCount: itemStock } }, upsert: true } });
                                }
                                else if (!serialNumber && !batchCode) {
                                    bulkOps.push({ updateOne: { filter: { branchId: branch._id }, update: { $inc: { stockCount: -itemStock, stockOutCount: itemStock }, $set: { stockType: job_order_1.stockTypes.out } }, } });
                                    bulkOps.push({ updateOne: { filter: { branchId: checkBranch._id }, update: { $inc: { stockCount: itemStock }, $set: { branchCode: checkBranch.branchCode, itemCode: itemsDetail.itemCode } }, upsert: true } });
                                }
                            }
                        }
                        if ((0, lodash_1.size)(bulkOps) > 0) {
                            await this.productsSerialNumberModel.bulkWrite(bulkOps, { ordered: false });
                        }
                    }
                    else if (usedItems && (0, lodash_1.size)(usedItems) > 0) {
                        let totalUsedItemStock = 0;
                        const bulkOps = [];
                        for (const usedItem of usedItems) {
                            const { serialNumber, batchCode, itemStock } = usedItem;
                            let filter = { branchId: branch._id };
                            if (serialNumber)
                                filter.serialNumber = serialNumber;
                            if (batchCode)
                                filter.batchNumber = batchCode;
                            const item = await this.productsSerialNumberModel.findOne(filter);
                            if (item) {
                                if ((!serialNumber && !batchCode && !item.serialNumber && !item.batchNumber) || (serialNumber && !batchCode) || (!serialNumber && batchCode)) {
                                    const newStockCount = Math.max(item.stockCount - itemStock, 0);
                                    let update = { $set: { stockCount: newStockCount, branchCode: branch.branchCode, itemCode: itemsDetail.itemCode }, $inc: { stockOutCount: itemStock } };
                                    if (serialNumber && !batchCode) {
                                        update.$set.stockType = job_order_1.stockTypes.out;
                                    }
                                    bulkOps.push({ updateOne: { filter: { _id: item._id }, update } });
                                    totalUsedItemStock -= itemStock;
                                }
                            }
                        }
                        if ((0, lodash_1.size)(bulkOps)) {
                            await this.productsSerialNumberModel.bulkWrite(bulkOps, { ordered: false });
                        }
                        if (totalUsedItemStock !== 0) {
                            stockUpdate.$set.itemStock = totalUsedItemStock;
                            totalProductStock = totalUsedItemStock;
                        }
                    }
                    else {
                        stockUpdate.$set.itemStock = itemStock;
                        stockUpdate.$set.branchCode = branch.branchCode;
                        stockUpdate.$set.itemCode = itemsDetail.itemCode;
                        totalProductStock = Number(itemStock);
                    }
                    stockInventoryBulkOperations.push({
                        updateOne: {
                            filter: { productId: itemsDetail._id, branchId: branch._id },
                            update: stockUpdate,
                            upsert: true,
                        },
                    });
                    addToBulkInsert(itemsDetail, branch, job_order_1.stockTypes.in, serialNumbers, batchCodes, itemStock);
                    productBulkOps.push({ updateOne: { filter: { itemCode }, update: { $inc: { itemQuantity: totalProductStock } } } });
                }
                if ((0, lodash_1.size)(productBulkOps)) {
                    await this.productsModel.bulkWrite(productBulkOps, { ordered: false });
                }
                if ((0, lodash_1.size)(stockInventoryBulkOperations) > 0) {
                    await this.productStockInventoryModel.bulkWrite(stockInventoryBulkOperations);
                }
                if ((0, lodash_1.size)(serialAndBatchBulkInsert) > 0) {
                    await this.productsSerialNumberModel.insertMany(serialAndBatchBulkInsert);
                }
                const stockAggregation = await this.productStockInventoryModel.aggregate([
                    {
                        $group: {
                            _id: "$productId",
                            totalItemStock: { $sum: "$itemStock" },
                        },
                    },
                ]);
                const bulkUpdateOperations = stockAggregation.map(stock => ({
                    updateOne: {
                        filter: { _id: stock._id },
                        update: { $set: { totalItemStock: stock.totalItemStock } }
                    }
                }));
                if ((0, lodash_1.size)(bulkUpdateOperations) > 0) {
                    await item_details_model_1.default.bulkWrite(bulkUpdateOperations);
                }
                if ((0, lodash_1.size)(errorMessages) > 0) {
                    return this.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: errorMessages,
                    });
                }
                else {
                    return this.sendSuccessResponse(res, {
                        message: 'Item detail stock updated successfully!',
                    }, 200);
                }
            }
            else {
                return this.sendErrorResponse(res, 200, {
                    message: 'Stock inventory is required!',
                });
            }
        }
        else {
            return this.sendErrorResponse(res, 200, {
                message: 'Validation error',
                validation: (0, helpers_1.formatZodError)(validatedData.error.errors),
            });
        }
    }
    async updateBomItemQuanity(req, res) {
        try {
            const productId = req.params.id;
            if (!productId) {
                return this.sendErrorResponse(res, 200, {
                    message: `Item master id is required`,
                });
            }
            const productMasterData = await items_master_schema_1.updateBomItemQuantitySchema.parseAsync(req.body);
            const itemMasterCodes = Array.from(new Set(productMasterData.map(item => item.itemMasterCode)));
            const existingItems = await items_master_model_1.default.find({ itemMasterCode: { $in: itemMasterCodes } }).select('_id itemType itemMasterCode bomMasterQuantity bomMasterOldQuantity itemMasterPrice').lean();
            const existingCodesSet = new Set(existingItems.map(item => item.itemMasterCode));
            const missingCodes = itemMasterCodes.filter(code => !existingCodesSet.has(code));
            if (missingCodes.length > 0) {
                return this.sendErrorResponse(res, 200, {
                    message: 'Some itemMasterCodes do not exist in the itemsMaster collection.',
                    missingItemMasterCodes: missingCodes,
                });
            }
            const updatedItemMasterCodes = [];
            for (let updateData of productMasterData) {
                const { itemMasterCode, bomMasterQuantity, itemMasterPrice } = updateData;
                const itemMasterDetail = existingItems.find((item) => item.itemMasterCode === itemMasterCode);
                if (!(0, lodash_1.isEmpty)(itemMasterDetail)) {
                    const updatedItemMasterData = {};
                    if (bomMasterQuantity && bomMasterQuantity > 0) {
                        updatedItemMasterData.bomMasterQuantity = bomMasterQuantity;
                        updatedItemMasterData.bomMasterOldQuantity = itemMasterDetail.bomMasterQuantity;
                    }
                    if (itemMasterPrice && itemMasterPrice > 0) {
                        updatedItemMasterData.itemMasterPrice = itemMasterPrice;
                    }
                    const updatedJobDetailData = await items_master_model_1.default.findOneAndUpdate({ _id: itemMasterDetail._id }, { $set: updatedItemMasterData });
                    updatedItemMasterCodes.push(itemMasterCode);
                    if (!(0, lodash_1.isEmpty)(updatedJobDetailData) && bomMasterQuantity && bomMasterQuantity > 0) {
                        const calculatedBomChangedPercentage = ((bomMasterQuantity - itemMasterDetail.bomMasterQuantity) / itemMasterDetail.bomMasterQuantity) * 100;
                        if (calculatedBomChangedPercentage !== 0) {
                            const itemDetails = await item_details_model_1.default.find({ productId: updatedJobDetailData._id });
                            if ((0, lodash_1.isArray)(itemDetails) && (0, lodash_1.size)(itemDetails) > 0) {
                                for (let itemDetail of itemDetails) {
                                    const newQuantity = itemDetail.itemTypeQuantity + (itemDetail.itemTypeQuantity * calculatedBomChangedPercentage / 100);
                                    const roundedQuantity = Math.max(0.999, Math.round(newQuantity * 1000) / 1000);
                                    console.log(itemDetail.itemTypeQuantity, 'itemTypeQuantity', roundedQuantity, newQuantity);
                                    await item_details_model_1.default.updateOne({ _id: itemDetail._id }, { $set: { itemTypeQuantity: roundedQuantity } });
                                }
                            }
                        }
                    }
                }
            }
            return this.sendSuccessResponse(res, {
                requestedData: updatedItemMasterCodes,
                message: 'Item master updation successfully completed!'
            });
        }
        catch (error) {
            if (error instanceof zod_1.ZodError) {
                return this.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(error.errors),
                });
            }
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    const errorDetail = errors[key];
                    validationErrors[key] = {
                        message: errorDetail.message,
                        value: errorDetail.value,
                    };
                });
                if (Object.keys(validationErrors).length > 0) {
                    return this.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors,
                    });
                }
            }
            return this.sendErrorResponse(res, 500, {
                message: error.message || 'An error occurred while creating the item master.',
            });
        }
    }
    async findOne(req, res) {
        try {
            const productId = req.params.id;
            if (productId) {
                const itemMaster = await this.inventoryService.findOne(productId);
                return this.sendSuccessResponse(res, {
                    requestedData: itemMaster,
                    message: 'Success'
                });
            }
            else {
                return this.sendErrorResponse(res, 200, {
                    message: 'Item master Id not found!',
                });
            }
        }
        catch (error) {
            return this.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async findOneItemMasterDetails(req, res) {
        try {
            const productId = req.params.id;
            if (productId) {
                let itemMaster = await this.inventoryService.findOne(productId);
                if ((0, lodash_1.isEmpty)(itemMaster)) {
                    return this.sendErrorResponse(res, 200, {
                        message: 'Item master details not found!',
                    });
                }
                if (!(0, lodash_1.isEmpty)(itemMaster.category)) {
                    let query = { _id: { $exists: true } };
                    query._id = itemMaster.category._id;
                    const categories = await category_service_1.default.findAllParentCategories({ query });
                    itemMaster = {
                        ...itemMaster,
                        category: categories
                    };
                }
                return this.sendSuccessResponse(res, {
                    requestedData: itemMaster,
                    message: 'Success'
                });
            }
            else {
                return this.sendErrorResponse(res, 200, {
                    message: 'Item master Id not found!',
                });
            }
        }
        catch (error) {
            return this.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    // public async update(req: Request, res: Response): Promise<void> {
    //     try {
    //         const productId = req.params.id;
    //         if (!productId) {
    //             return this.sendErrorResponse(res, 400, {
    //                 message: 'Item master Id not provided! Please try again with an item master Id.',
    //             });
    //         }
    //         const validatedData = await productSchema.safeParseAsync({ ...req.body, _id: productId });
    //         if (!validatedData.success) {
    //             return this.sendErrorResponse(res, 400, {
    //                 message: 'Validation error',
    //                 validation: formatZodError(validatedData.error.errors),
    //             });
    //         }
    //         const updatedItemMasterData = validatedData.data;
    //         const updatedItemMaster = await this.inventoryService.update(productId, { ...updatedItemMasterData, updatedAt: new Date() });
    //         if (updatedItemMaster) {
    //             return this.sendSuccessResponse(res, {
    //                 requestedData: updatedItemMaster,
    //                 message: 'Item master updated successfully!',
    //             }, 200, { // task log
    //                 sourceFromId: updatedItemMaster._id,
    //                 sourceFrom: adminTaskLog.inventoryManagement.items.itemMaster,
    //                 activity: adminTaskLogActivity.update,
    //                 activityStatus: adminTaskLogStatus.success,
    //                 activityComment: 'Item master updated successfully',
    //             });
    //         } else {
    //             return this.sendErrorResponse(res, 404, {
    //                 message: 'Item master Id not found!',
    //             });
    //         }
    //     } catch (error: any) {
    //         return this.sendErrorResponse(res, 500, {
    //             message: error.message || 'Some error occurred while updating item master',
    //         });
    //     }
    // }
    async statusChange(req, res) {
        try {
            const validatedData = items_master_schema_1.itemMasterStatusSchema.safeParse(req.body);
            if (validatedData.success) {
                const productId = req.params.id;
                if (productId) {
                    let { status } = req.body;
                    const updateditemMasterData = { status };
                    const updateditemMaster = await this.inventoryService.update(productId, updateditemMasterData);
                    if (updateditemMaster) {
                        return this.sendSuccessResponse(res, {
                            requestedData: updateditemMaster,
                            message: 'Item master status updated successfully!'
                        });
                    }
                    else {
                        return this.sendErrorResponse(res, 200, {
                            message: 'itemMaster Id not found!',
                        }, req);
                    }
                }
                else {
                    return this.sendErrorResponse(res, 200, {
                        message: 'Item master Id not found! Please try again with Item master id',
                    }, req);
                }
            }
            else {
                return this.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                }, req);
            }
        }
        catch (error) {
            return this.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating Item master'
            }, req);
        }
    }
    async destroy(req, res) {
        try {
            const productId = req.params.id;
            if (productId) {
                const itemMaster = await this.inventoryService.findOne(productId);
                if (itemMaster) {
                    // await ProductsInventoryService.destroy(productId);
                    // return this.sendSuccessResponse(res, { message: 'ItemMaster deleted successfully!' });
                    return this.sendErrorResponse(res, 200, {
                        message: 'You cant delete this Item master',
                    });
                }
                else {
                    return this.sendErrorResponse(res, 200, {
                        message: 'This item master details not found!',
                    });
                }
            }
            else {
                return this.sendErrorResponse(res, 200, {
                    message: 'Item master id not found!',
                });
            }
        }
        catch (error) {
            return this.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting item master' });
        }
    }
}
exports.default = ProductsController;
