"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../base-controller"));
const warranty_service_1 = __importDefault(require("../../../services/admin/inventory-management/warranty-service"));
const warranty_schema_1 = require("../../../utils/schemas/admin/inventory-management/warranty-schema");
const zod_1 = require("zod");
const controller = new base_controller_1.default();
class WarrantyController extends base_controller_1.default {
    async findAll(req, res) {
        const { page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { warrantyTitle: keywordRegex }
                ],
                ...query
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const warranty = await warranty_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: warranty,
                totalCount: await warranty_service_1.default.getTotalCount(query),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching warranty' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = await warranty_schema_1.warrantySchema.parseAsync(req.body);
            const { warrantyTitle, warrantyCode, slug, warrantyPeriod, termsAndConditions, coverageDetails, status } = validatedData;
            const user = res.locals.user;
            const warrantyData = {
                warrantyTitle,
                slug: (0, helpers_1.slugify)(warrantyTitle),
                warrantyCode,
                warrantyPeriod,
                termsAndConditions,
                coverageDetails,
                status: status || 'active',
                createdBy: user._id,
                createdAt: new Date(),
                updatedAt: new Date(),
            };
            const newWarranty = await warranty_service_1.default.create(warrantyData);
            return controller.sendSuccessResponse(res, {
                requestedData: newWarranty,
                message: 'Warranty created successfully!',
            }, 200, {
                sourceFromId: newWarranty._id,
                sourceFrom: task_log_1.adminTaskLog.inventoryManagement.warranty,
                activity: task_log_1.adminTaskLogActivity.create,
                activityStatus: task_log_1.adminTaskLogStatus.success,
                activityComment: "Warranty created successfully"
            });
        }
        catch (error) {
            if (error instanceof zod_1.ZodError) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(error.errors),
                });
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'An error occurred while creating the warranty.',
            });
        }
    }
    async findOne(req, res) {
        try {
            const warrantyId = req.params.id;
            if (warrantyId) {
                const warranty = await warranty_service_1.default.findOne(warrantyId);
                return controller.sendSuccessResponse(res, {
                    requestedData: warranty,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Warranty Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = warranty_schema_1.warrantySchema.safeParse(req.body);
            if (validatedData.success) {
                const warrantyId = req.params.id;
                if (warrantyId) {
                    const updatedWarrantyData = req.body;
                    const updatedWarranty = await warranty_service_1.default.update(warrantyId, { ...updatedWarrantyData, updatedAt: new Date() });
                    if (updatedWarranty) {
                        return controller.sendSuccessResponse(res, {
                            requestedData: updatedWarranty,
                            message: 'Warranty updated successfully!'
                        }, 200, {
                            sourceFromId: updatedWarranty._id,
                            sourceFrom: task_log_1.adminTaskLog.inventoryManagement.warranty,
                            activity: task_log_1.adminTaskLogActivity.update,
                            activityStatus: task_log_1.adminTaskLogStatus.success,
                            activityComment: "Warranty updated successfully"
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'Warranty Id not found!',
                        });
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Warranty Id not found! Please try again with Warranty id',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating warranty'
            });
        }
    }
    async destroy(req, res) {
        try {
            const warrantyId = req.params.id;
            if (warrantyId) {
                const warranty = await warranty_service_1.default.findOne(warrantyId);
                if (warranty) {
                    // await WarrantyService.destroy(warrantyId);
                    // return controller.sendSuccessResponse(res, { message: 'Warranty deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this warranty',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This warranty details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Warranty id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting warranty' });
        }
    }
}
exports.default = new WarrantyController();
