"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const zod_1 = require("zod");
const lodash_1 = require("lodash");
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../base-controller"));
const job_order_service_1 = __importDefault(require("../../../services/admin/job-order/job-order-service"));
const job_order_schema_1 = require("../../../utils/schemas/admin/job-order/job-order-schema");
const job_order_1 = require("../../../constants/admin/job-order");
const job_order_details_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-details-model"));
const job_order_master_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-master-model"));
const job_order_assignment_history_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-assignment-history-model"));
const job_order_used_items_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-used-items-model"));
const job_order_item_attributes_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-item-attributes-model"));
const job_order_gallery_image_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-gallery-image-model"));
const spare_requests_and_transfers_model_1 = __importDefault(require("../../../model/admin/inventory-management/items/spare-requests-and-transfers-model"));
const items_1 = require("../../../constants/items");
const sequence_model_1 = __importDefault(require("../../../model/general/sequence-model"));
const project_utils_1 = require("../../../utils/project-utils");
const job_order_customer_item_history_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-customer-item-history-model"));
const branch_model_1 = __importDefault(require("../../../model/admin/operations/branch-model"));
const admin_user_model_1 = __importDefault(require("../../../model/admin/staff-management/admin-user-model"));
const job_order_sale_employees_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-sale-employees-model"));
const job_order_by_products_1 = __importDefault(require("../../../model/admin/job-order/job-order-by-products"));
const category_config_1 = require("../../../utils/config/admin/inventory-management/category-config");
const category_model_1 = __importDefault(require("../../../model/admin/inventory-management/category-model"));
const category_1 = require("../../../utils/admin/category");
const conditions_model_1 = __importDefault(require("../../../model/admin/other/conditions-model"));
const complaint_model_1 = __importDefault(require("../../../model/admin/inventory-management/complaint-model"));
const additional_cost_values_model_1 = __importDefault(require("../../../model/admin/other/additional-cost-values-model"));
const controller = new base_controller_1.default();
class JobOrderController extends base_controller_1.default {
    async findAll(req, res) {
        const { jobOrderStatus, customerId = '', branchId = '', fromDate, endDate, page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        // if (status && status !== '') {
        //     query.status = { $in: Array.isArray(status) ? status : [status] };
        // } else {
        //     query.status = '1';
        // }
        if (keyword) {
            const escapedKeyword = keyword.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
            const keywordRegex = new RegExp(`^${keyword}`, 'i');
            query = {
                $or: [
                    { jobOrderTitle: { $regex: keywordRegex } },
                    { 'jobId': { $regex: keywordRegex } },
                    { 'customer.customerTitle': { $regex: keywordRegex } },
                    { 'customer.customerEmail': { $regex: keywordRegex } },
                    { 'customer.customerCode': { $regex: keywordRegex } },
                    { 'customer.customerPhoneNumber': { $regex: keywordRegex } },
                    { 'customer.customerCountryCode': { $regex: keywordRegex } },
                    { 'branch.branchTitle': { $regex: keywordRegex } },
                    { 'branch.branchCode': { $regex: keywordRegex } },
                ],
                ...query
            };
        }
        if (branchId) {
            if (query.$or) {
                query.$or = [
                    ...query.$or,
                    { createdBranchId: new mongoose_1.default.Types.ObjectId(branchId) },
                    { 'jobOrderDetail.transferBranchId': new mongoose_1.default.Types.ObjectId(branchId) },
                    { 'jobOrderDetail.branchId': new mongoose_1.default.Types.ObjectId(branchId) },
                ];
            }
            else {
                query.$or = [
                    { createdBranchId: new mongoose_1.default.Types.ObjectId(branchId) },
                    { 'jobOrderDetail.transferBranchId': new mongoose_1.default.Types.ObjectId(branchId) },
                    { 'jobOrderDetail.branchId': new mongoose_1.default.Types.ObjectId(branchId) },
                ];
            }
        }
        if (customerId) {
            query.customerId = new mongoose_1.default.Types.ObjectId(customerId);
        }
        if (jobOrderStatus) {
            query.jobOrderStatus = jobOrderStatus;
        }
        if (fromDate || endDate) {
            const dateFilter = {};
            if (fromDate) {
                dateFilter.$gte = (0, helpers_1.dateConvertPm)(fromDate);
            }
            if (endDate) {
                dateFilter.$lte = (0, helpers_1.dateConvertPm)(endDate);
            }
            if (Object.keys(dateFilter).length > 0) {
                query.jobOrderStatusAt = dateFilter;
            }
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const jobOrder = await job_order_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: jobOrder,
                totalCount: await job_order_service_1.default.getTotalCount(query),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching jobOrder' });
        }
    }
    async findJobOrderDetails(req, res) {
        try {
            const { jobOrderId = '', jobOrderDetailId = '', branchId = '', technicianId = '', serviceStatus = '', technicianStatus = '', assignTasks = '', page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
            let query = { _id: { $exists: true } };
            if (jobOrderId) {
                query.jobOrderId = new mongoose_1.default.Types.ObjectId(jobOrderId);
            }
            if (branchId) {
                query.branchId = new mongoose_1.default.Types.ObjectId(branchId);
            }
            if (jobOrderDetailId) {
                query.jobOrderDetailId = new mongoose_1.default.Types.ObjectId(jobOrderDetailId);
            }
            if (technicianId) {
                query.technicianId = new mongoose_1.default.Types.ObjectId(technicianId);
            }
            if (serviceStatus) {
                query.serviceStatus = serviceStatus;
            }
            if (assignTasks) {
                query.serviceStatus = { $ne: job_order_1.serviceStatusListJson.completed };
            }
            if (technicianStatus) {
                query.technicianStatus = technicianStatus;
            }
            const sort = {};
            if (sortby && sortorder) {
                sort[sortby] = sortorder === 'desc' ? -1 : 1;
            }
            const [jobOrderDetails] = await job_order_service_1.default.findJobOrderDetails({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: jobOrderDetails.data,
                totalCount: jobOrderDetails.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async findAllJobOrderAssignmentHistory(req, res) {
        try {
            const { jobOrderId = '', jobOrderDetailId = '', technicianId = '', customerId = '', branchId = '', page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
            let query = { _id: { $exists: true } };
            if (jobOrderId) {
                query.jobOrderId = new mongoose_1.default.Types.ObjectId(jobOrderId);
            }
            if (jobOrderDetailId) {
                query.jobOrderDetailId = new mongoose_1.default.Types.ObjectId(jobOrderDetailId);
            }
            if (technicianId) {
                query.technicianId = new mongoose_1.default.Types.ObjectId(technicianId);
            }
            const sort = {};
            if (sortby && sortorder) {
                sort[sortby] = sortorder === 'desc' ? -1 : 1;
            }
            const jobOrder = await job_order_service_1.default.findAllJobOrderAssignmentHistory({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: jobOrder,
                totalCount: await job_order_service_1.default.getTotalCount(query),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async findJobOrderDatas(req, res) {
        const { categoryId, isComplaints = '', isConditions = '', isCostValues = '', page_size = 1, limit = 10, status = ['0', '1', '2'], } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        let categoryIds = [];
        if (categoryId) {
            const categoryBatchIds = (0, lodash_1.uniq)([categoryId]).map(id => new mongoose_1.default.Types.ObjectId(id));
            const categoriesWithChilleds = await category_model_1.default.aggregate((0, category_config_1.categoriesWithChildrenInArrayAggregation)(categoryBatchIds));
            categoryIds = (0, category_1.extractAllCategoryIds)(categoriesWithChilleds);
        }
        try {
            const requestedData = {};
            if (isConditions === '1') {
                (0, lodash_1.set)(requestedData, 'conditions', await conditions_model_1.default.find({ categoryId: { $in: categoryIds } }));
            }
            if (isComplaints === '1') {
                (0, lodash_1.set)(requestedData, 'complaints', await complaint_model_1.default.find({ categoryId: { $in: categoryIds } }));
            }
            if (isCostValues === '1') {
                (0, lodash_1.set)(requestedData, 'additionalCostValues', await additional_cost_values_model_1.default.find());
            }
            return controller.sendSuccessResponse(res, {
                requestedData,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching condition' });
        }
    }
    async findAllJobOrderCustomerItemTransferHistory(req, res) {
        try {
            const { jobOrderId = '', jobOrderDetailId = '', technicianId = '', driverId = '', customerId = '', branchId = '', page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
            let query = { _id: { $exists: true } };
            const user = await res.locals.user;
            const visitorBranchId = (0, project_utils_1.getVisitorBranchId)(user);
            if (branchId) {
                query.$or = [
                    { 'branchId': new mongoose_1.default.Types.ObjectId(branchId) },
                    { 'transferFromBranchId': new mongoose_1.default.Types.ObjectId(branchId) }
                ];
            }
            else if (visitorBranchId) {
                query.$or = [
                    { 'branchId': new mongoose_1.default.Types.ObjectId(visitorBranchId) },
                    { 'transferFromBranchId': new mongoose_1.default.Types.ObjectId(visitorBranchId) }
                ];
            }
            if (keyword) {
                const keywordRegex = new RegExp(`^${keyword}`, 'i');
                const jobOrderQuery = { 'jobOrder.jobId': Number(keyword) };
                query.$or.push({ 'jobOrder.jobId': Number(keyword) }, { 'driver.userEmail': { $regex: keywordRegex } }, { 'driver.userCode': { $regex: keywordRegex } }, { 'driver.userTitle': { $regex: keywordRegex } }, { 'customer.customerCode': { $regex: keywordRegex } }, { 'customer.customerTitle': { $regex: keywordRegex } }, { 'customer.customerPhoneNumber': { $regex: keywordRegex } }, { 'transferFromBranch.branchCode': { $regex: keywordRegex } }, { 'transferFromBranch.branchTitle': { $regex: keywordRegex } });
            }
            if (driverId) {
                query.driverId = new mongoose_1.default.Types.ObjectId(driverId);
            }
            if (jobOrderId) {
                query.jobOrderMasterId = new mongoose_1.default.Types.ObjectId(jobOrderId);
            }
            if (jobOrderDetailId) {
                query.jobOrderDetailId = new mongoose_1.default.Types.ObjectId(jobOrderDetailId);
            }
            if (customerId) {
                query.customerId = new mongoose_1.default.Types.ObjectId(customerId);
            }
            const sort = {};
            if (sortby && sortorder) {
                sort[sortby] = sortorder === 'desc' ? -1 : 1;
            }
            const jobOrder = await job_order_service_1.default.findAllJobOrderCustomerItemTransferHistory({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: jobOrder.data,
                totalCount: jobOrder.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async create(req, res) {
        try {
            const validatedData = await job_order_schema_1.createJobOrderSchema.parseAsync(req.body);
            let { jobOrderId, createdBranchId, saleEmployeeId, transferBranchId, selectedItemDetails, itemMasterId, parentCategoryId, categoryId, customerId, complaintId, vendorId, conditionId, paymentTermId, costCenterId, additionalFields, dueDate, serviceType, costItems, customerSignature, totalAmount, balanceAmount, subtotalAmount, attributes, customerComments, remarks, serialNumber, assignmentHistory, saleEmployees } = validatedData;
            const user = res.locals.user;
            let jobOrderDetails = null;
            if (!jobOrderId) {
                const jobOrderData = {
                    customerId,
                    createdBranchId,
                    jobOrderStatus: job_order_1.jobOrderStatusListJson.pending,
                    jobOrderStatusAt: new Date(),
                    createdBy: user._id,
                    createdAt: new Date(),
                    updatedAt: new Date(),
                };
                jobOrderDetails = await job_order_service_1.default.create(jobOrderData);
                if ((0, lodash_1.isEmpty)(jobOrderDetails)) {
                    return controller.sendErrorResponse(res, 500, {
                        message: 'Something went wrong while creating job order',
                    });
                }
                jobOrderId = jobOrderDetails._id;
            }
            if (!jobOrderId) {
                return controller.sendErrorResponse(res, 500, {
                    message: 'Something went wrong while creating job order',
                });
            }
            else {
                jobOrderDetails = await job_order_master_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(jobOrderId) });
                if ((0, lodash_1.isEmpty)(jobOrderDetails)) {
                    return controller.sendErrorResponse(res, 500, {
                        message: 'This job order not found',
                    });
                }
            }
            const totalQuantity = (0, lodash_1.size)(selectedItemDetails) > 0 ? selectedItemDetails.reduce((acc, item) => acc + item.quantity, 0) : 0;
            const jobOrdeDetailsData = {
                jobMasterId: jobOrderId,
                branchId: transferBranchId ? transferBranchId : createdBranchId,
                transferBranchId: transferBranchId ? createdBranchId : null,
                saleEmployeeId,
                categoryId,
                parentCategoryId,
                itemMasterId,
                complaintId,
                vendorId: vendorId !== '' ? vendorId : null,
                conditionId: conditionId !== '' ? conditionId : null,
                costCenterId: costCenterId !== '' ? costCenterId : null,
                paymentTermId: paymentTermId !== '' ? paymentTermId : null,
                additionalFields,
                itemMasterAttributes: (0, lodash_1.size)(attributes) > 0 ? attributes.map((attribute) => ({
                    attributeId: new mongoose_1.default.Types.ObjectId(attribute.attributeId),
                    attributeDetailId: new mongoose_1.default.Types.ObjectId(attribute.attributeDetailId),
                })) : [],
                costItems,
                totalQuantity,
                subtotalAmount,
                balanceAmount,
                totalAmount,
                customerSignature,
                serialNumber: Number(serialNumber),
                remarks,
                customerComments,
                serviceType,
                serviceStatus: job_order_1.serviceStatusListJson.pending,
                serviceStatusAt: new Date(),
                technicianStatus: job_order_1.technicianStatusListJson.pending,
                technicianStatusAt: new Date(),
                dueDate: dueDate ?? null,
                createdBy: user._id,
                createdAt: new Date(),
                updatedAt: new Date(),
            };
            const newJobOrderDetails = await job_order_details_model_1.default.create(jobOrdeDetailsData);
            console.log(JSON.stringify(newJobOrderDetails, null, 2));
            if ((0, lodash_1.isEmpty)(newJobOrderDetails)) {
                await job_order_master_model_1.default.findOneAndDelete({ _id: jobOrderId });
                return controller.sendErrorResponse(res, 500, {
                    message: 'Something went wrong while creating job order detail',
                });
            }
            await job_order_customer_item_history_model_1.default.create({
                jobOrderMasterId: jobOrderId,
                customerId,
                jobOrderDetailId: newJobOrderDetails._id,
                branchId: createdBranchId,
                historyType: job_order_1.jobOrderCustomerItemTransferHistoryJson.created,
                serviceStatusMessage: `Job order ${jobOrderDetails.jobId} successfully created.`
            });
            if (transferBranchId) {
                await job_order_customer_item_history_model_1.default.create({
                    customerId,
                    jobOrderMasterId: jobOrderId,
                    jobOrderDetailId: newJobOrderDetails._id,
                    branchId: transferBranchId,
                    transferFromBranchId: createdBranchId,
                    historyType: job_order_1.jobOrderCustomerItemTransferHistoryJson.transfer,
                    historyStatus: job_order_1.customerItemTransferStatusJson.initiated,
                });
            }
            if ((0, lodash_1.size)(attributes) > 0) {
                const attributesInsertData = attributes
                    .filter((attribute) => attribute.attributeId && attribute.attributeDetailId)
                    .map((attribute) => ({
                    jobMasterId: jobOrderId,
                    jobOrderDetailId: newJobOrderDetails._id,
                    itemMasterId: new mongoose_1.default.Types.ObjectId(itemMasterId),
                    attributeId: new mongoose_1.default.Types.ObjectId(attribute.attributeId),
                    attributeDetailId: new mongoose_1.default.Types.ObjectId(attribute.attributeDetailId),
                    createdBy: user._id,
                }));
                if (attributesInsertData.length > 0) {
                    await job_order_item_attributes_model_1.default.insertMany(attributesInsertData);
                }
            }
            if (!(0, lodash_1.isEmpty)(assignmentHistory) && (0, lodash_1.size)(assignmentHistory) > 0) {
                const sequenceDoc = await sequence_model_1.default.findOneAndUpdate({ _id: 'taskSequence' }, { $inc: { sequenceValue: assignmentHistory.length } }, { new: true, upsert: true });
                if (sequenceDoc) {
                    const startingTaskId = sequenceDoc.sequenceValue - assignmentHistory.length + 1;
                    const assignmentHistoryInsertData = assignmentHistory
                        .filter((history) => history.technicianId !== '')
                        .map((history, index) => ({
                        jobMasterId: jobOrderId,
                        jobOrderDetailId: newJobOrderDetails._id,
                        branchId: createdBranchId,
                        technicianId: history.technicianId,
                        createdBy: user._id,
                        taskId: startingTaskId + index,
                    }));
                    if (assignmentHistoryInsertData.length > 0) {
                        await job_order_assignment_history_model_1.default.insertMany(assignmentHistoryInsertData);
                        console.log('Assignment history inserted with task IDs:', assignmentHistoryInsertData);
                    }
                }
                else {
                    console.error('Failed to generate task IDs.');
                }
            }
            if ((0, lodash_1.size)(selectedItemDetails) > 0) {
                const validItems = selectedItemDetails.filter((item) => item.itemId && item.itemId !== '');
                const usedItemsInsertData = validItems.map((item) => {
                    const validQuantity = (0, lodash_1.isNumber)(item.quantity) && item.quantity > 0;
                    return {
                        jobMasterId: jobOrderId,
                        jobOrderDetailId: newJobOrderDetails._id,
                        itemMasterId: new mongoose_1.default.Types.ObjectId(itemMasterId),
                        itemId: new mongoose_1.default.Types.ObjectId(item.itemId),
                        technicianId: ((0, lodash_1.size)(assignmentHistory) > 0 && assignmentHistory.some((assignment) => assignment.technicianId !== '')) ? assignmentHistory[0].technicianId : null,
                        quantity: validQuantity ? item.quantity : 0,
                        itemPrice: item.itemPrice,
                        buyedItemPrice: item.totalPrice,
                        itemTypeQuantity: item.itemTypeQuantity,
                        itemType: item.itemType,
                        createdBy: user._id,
                    };
                });
                if (usedItemsInsertData.length > 0) {
                    await job_order_used_items_model_1.default.insertMany(usedItemsInsertData);
                }
            }
            if (!(0, lodash_1.isEmpty)(saleEmployees) && (0, lodash_1.size)(saleEmployees) > 0) {
                const saleEmployeesInsertData = saleEmployees
                    .filter((history) => history.technicianId !== '')
                    .map((history, index) => ({
                    jobMasterId: jobOrderId,
                    jobOrderDetailId: newJobOrderDetails._id,
                    branchId: createdBranchId,
                    saleEmployeeId: history.saleEmployeeId,
                    createdBy: user._id,
                }));
                if (saleEmployeesInsertData.length > 0) {
                    await job_order_sale_employees_model_1.default.insertMany(saleEmployeesInsertData);
                    console.log('Sale employees IDs:', saleEmployeesInsertData);
                }
            }
            const galleryImage = req.files.filter((file) => file.fieldname &&
                file.fieldname.startsWith('galleryImage'));
            if ((0, lodash_1.isArray)(galleryImage) && (0, lodash_1.size)(galleryImage) > 0) {
                const documents = await Promise.all(galleryImage.map(async (file) => {
                    const imageUrl = await (0, helpers_1.handleFileUpload)(req, null, file, 'galleryImageUrl', 'galleryimages');
                    return {
                        jobOrderMasterId: jobOrderId,
                        jobOrderDetailId: newJobOrderDetails._id,
                        galleryImageUrl: imageUrl,
                        type: job_order_1.galleryImageTypes.service,
                        createdAt: new Date()
                    };
                }));
                await job_order_gallery_image_model_1.default.insertMany(documents);
            }
            const customerIdGalleryImage = req.files.filter((file) => file.fieldname &&
                file.fieldname.startsWith('customerIdGalleryImage'));
            if ((0, lodash_1.isArray)(customerIdGalleryImage) && (0, lodash_1.size)(customerIdGalleryImage) > 0) {
                const documents = await Promise.all(customerIdGalleryImage.map(async (file) => {
                    const imageUrl = await (0, helpers_1.handleFileUpload)(req, null, file, 'customerIdImageUrl', 'galleryimages');
                    return {
                        jobOrderMasterId: jobOrderId,
                        jobOrderDetailId: newJobOrderDetails._id,
                        galleryImageUrl: imageUrl,
                        type: job_order_1.galleryImageTypes.customerId,
                        createdAt: new Date()
                    };
                }));
                await job_order_gallery_image_model_1.default.insertMany(documents);
            }
            const [result] = await job_order_details_model_1.default.aggregate([
                { $match: { jobMasterId: new mongoose_1.default.Types.ObjectId(jobOrderId) } },
                {
                    $group: {
                        _id: null,
                        totalQuantity: { $sum: { $ifNull: ["$totalQuantity", 0] } },
                        totalSubtotal: { $sum: { $ifNull: ["$subtotalAmount", 0] } },
                        totalBalance: { $sum: { $ifNull: ["$balanceAmount", 0] } },
                        totalAmountSum: { $sum: { $ifNull: ["$totalAmount", 0] } },
                    }
                }
            ]);
            let enrichedCostItems = null;
            if ((0, lodash_1.size)(costItems) > 0) {
                console.log('costItems', costItems);
                enrichedCostItems = costItems
                    .filter((costItem) => Number(costItem.costAmount) > 0)
                    .map((item) => ({
                    ...item,
                    jobOrderDetailId: newJobOrderDetails._id,
                }));
            }
            await job_order_master_model_1.default.findOneAndUpdate({ _id: jobOrderId }, {
                $set: {
                    jobTotalQuantity: result.totalQuantity,
                    jobSubTotalAmount: result.totalSubtotal,
                    jobBalanceAmount: result.totalBalance,
                    jobTotalAmount: result.totalAmountSum,
                    ...((0, lodash_1.size)(enrichedCostItems) > 0
                        ? {
                            costItems: jobOrderId
                                ? [...(jobOrderDetails.costItems || []), ...enrichedCostItems]
                                : [...enrichedCostItems],
                        }
                        : {}),
                    updatedAt: new Date(),
                },
            });
            return controller.sendSuccessResponse(res, {
                requestedData: jobOrderDetails,
                message: 'Job order created successfully!',
            }, 200, {
                sourceFromId: jobOrderId,
                sourceFromReferenceId: newJobOrderDetails._id,
                sourceFrom: task_log_1.adminTaskLog.jobOrder.jobOrder,
                activity: task_log_1.adminTaskLogActivity.create,
                activityStatus: task_log_1.adminTaskLogStatus.success,
                activityComment: "Job order created successfully"
            });
        }
        catch (error) {
            if (error instanceof zod_1.ZodError) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(error.errors),
                });
            }
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    validationErrors[key] = errors[key].message;
                });
                if (Object.keys(validationErrors).length > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors
                    });
                }
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'An error occurred while creating the jobOrder.',
            });
        }
    }
    async manageSpare(req, res) {
        try {
            const validatedData = job_order_schema_1.addSpareJobOrderSchema.safeParse(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
            let { orderDetailId, technicianId, selectedItemDetails, } = validatedData.data;
            const user = res.locals.user;
            const orderDetails = await job_order_details_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(orderDetailId) });
            if ((0, lodash_1.isEmpty)(orderDetails)) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Job order not found!',
                });
            }
            if (orderDetails.serviceStatus === job_order_1.serviceStatusListJson.completed) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'This Job order already completed!',
                });
            }
            const existingRecords = await job_order_used_items_model_1.default.find({
                jobMasterId: orderDetails.jobMasterId,
                jobOrderDetailId: new mongoose_1.default.Types.ObjectId(orderDetailId),
                technicianId: new mongoose_1.default.Types.ObjectId(technicianId),
            }).lean();
            for (const item of selectedItemDetails) {
                const filter = {
                    jobMasterId: orderDetails.jobMasterId,
                    jobOrderDetailId: new mongoose_1.default.Types.ObjectId(orderDetailId),
                    technicianId: new mongoose_1.default.Types.ObjectId(technicianId),
                };
                if (item.usedItemId) {
                    filter._id = new mongoose_1.default.Types.ObjectId(item.usedItemId);
                }
                else {
                    filter.itemId = new mongoose_1.default.Types.ObjectId(item.itemId);
                }
                const updateFields = {
                    jobMasterId: orderDetails.jobMasterId,
                    jobOrderDetailId: orderDetailId,
                    technicianId: technicianId,
                    createdBy: user._id,
                    itemId: item.itemId,
                    quantity: item.quantity,
                    itemTypeQuantity: item.itemTypeQuantity,
                    itemType: item.itemType,
                    itemPrice: item.itemPrice,
                    buyedItemPrice: item.quantity * item.itemPrice,
                };
                if (item.usedItemId) {
                    await job_order_used_items_model_1.default.updateOne(filter, { $set: updateFields });
                }
                else {
                    await job_order_used_items_model_1.default.updateOne(filter, { $set: updateFields }, { upsert: true });
                }
            }
            const updatedIds = selectedItemDetails
                .map((item) => item.usedItemId)
                .filter((id) => id);
            const idsToDelete = existingRecords
                .filter(record => !updatedIds.includes(record._id.toString()))
                .map(record => record._id);
            if (idsToDelete.length > 0) {
                await job_order_used_items_model_1.default.deleteMany({
                    _id: { $in: idsToDelete }
                });
            }
            const jobOrder = await job_order_service_1.default.findOneSimple({
                jobDetailQuery: {
                    $or: [
                        { 'transferBranchId': orderDetails?.transferBranchId },
                        { 'branchId': orderDetails?.branchId },
                    ],
                },
                jobOrderId: orderDetails.jobMasterId, simple: '0', jobdetails: '1', getcustomer: '1',
            });
            return controller.sendSuccessResponse(res, {
                requestedData: jobOrder,
                message: 'Items updated/inserted successfully',
            });
        }
        catch (error) {
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    validationErrors[key] = errors[key].message;
                });
                if (Object.keys(validationErrors).length > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors
                    });
                }
            }
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async manageByProduct(req, res) {
        try {
            const validatedData = job_order_schema_1.manageByProductJobOrderSchema.safeParse(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
            let { orderDetailId, itemMasterId, selectedItemDetails, } = validatedData.data;
            const user = res.locals.user;
            const orderDetails = await job_order_details_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(orderDetailId) });
            if ((0, lodash_1.isEmpty)(orderDetails)) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Job order not found!',
                });
            }
            if (orderDetails.serviceStatus === job_order_1.serviceStatusListJson.completed) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'This Job order already completed!',
                });
            }
            const existingRecords = await job_order_by_products_1.default.find({
                jobMasterId: orderDetails.jobMasterId,
                jobOrderDetailId: new mongoose_1.default.Types.ObjectId(orderDetailId),
            }).lean();
            const updatedIds = selectedItemDetails
                .map((item) => item.byProductId)
                .filter((id) => id);
            const idsToDelete = existingRecords
                .filter(record => !updatedIds.includes(record._id.toString()))
                .map(record => record._id);
            if (idsToDelete.length > 0) {
                await job_order_by_products_1.default.deleteMany({
                    _id: { $in: idsToDelete }
                });
            }
            for (const item of selectedItemDetails) {
                const filter = {
                    jobMasterId: orderDetails.jobMasterId,
                    jobOrderDetailId: new mongoose_1.default.Types.ObjectId(orderDetailId),
                };
                if (item.byProductId) {
                    filter._id = new mongoose_1.default.Types.ObjectId(item.byProductId);
                }
                else {
                    filter.itemId = new mongoose_1.default.Types.ObjectId(item.itemId);
                }
                const updateFields = {
                    jobMasterId: orderDetails.jobMasterId,
                    jobOrderDetailId: orderDetailId,
                    createdBy: user._id,
                    itemMasterId,
                    itemId: item.itemId,
                    quantity: item.quantity,
                };
                if (item.byProductId) {
                    await job_order_by_products_1.default.updateOne(filter, { $set: updateFields });
                }
                else {
                    await job_order_by_products_1.default.updateOne(filter, { $set: updateFields }, { upsert: true });
                }
            }
            const jobOrder = await job_order_service_1.default.findOneSimple({
                jobDetailQuery: {
                    $or: [
                        { 'transferBranchId': orderDetails?.transferBranchId },
                        { 'branchId': orderDetails?.branchId },
                    ],
                },
                jobOrderId: orderDetails.jobMasterId, simple: '0', jobdetails: '1', getcustomer: '1',
            });
            return controller.sendSuccessResponse(res, {
                requestedData: jobOrder,
                message: 'Items updated/inserted successfully',
            });
        }
        catch (error) {
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    validationErrors[key] = errors[key].message;
                });
                if (Object.keys(validationErrors).length > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors
                    });
                }
            }
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async spareRequestAndTransfer(req, res) {
        try {
            const validatedData = job_order_schema_1.spareRequestAndTransferJobOrderSchema.safeParse(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
            let { type, orderDetailId, itemMasterId, itemId, currentBranchId, newBranchId, quantity, priority, reason, remarks, requestedBy, transferredBy } = validatedData.data;
            const user = res.locals.user;
            const orderDetails = await job_order_details_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(orderDetailId) });
            if ((0, lodash_1.isEmpty)(orderDetails)) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Job order not found!',
                });
            }
            const allExistingRequest = await spare_requests_and_transfers_model_1.default.findOne({
                $and: [
                    { jobMasterId: orderDetails.jobMasterId },
                    { jobOrderDetailId: orderDetailId ? new mongoose_1.default.Types.ObjectId(orderDetailId) : null },
                    { itemMasterId: new mongoose_1.default.Types.ObjectId(itemMasterId) },
                    { itemId: new mongoose_1.default.Types.ObjectId(itemId) },
                    { currentBranchId: new mongoose_1.default.Types.ObjectId(currentBranchId) },
                    { newBranchId: new mongoose_1.default.Types.ObjectId(newBranchId) },
                    { requestedStatus: items_1.itemRequestedStatus.pending }
                ]
            });
            if (allExistingRequest) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'This item has already been requested from this branch. Please wait for the current request to be processed before trying again.',
                });
            }
            const isertData = {
                type,
                jobMasterId: orderDetails.jobMasterId,
                jobOrderDetailId: new mongoose_1.default.Types.ObjectId(orderDetailId),
                orderDetailId,
                itemMasterId,
                itemId,
                currentBranchId,
                newBranchId,
                quantity,
                priority,
                reason,
                remarks,
                ...(requestedBy ? {
                    requestedBy,
                    requestedAt: new Date(),
                    requestedStatus: items_1.itemRequestedStatus.pending
                } : { requestedBy: null }),
                transferredBy: transferredBy ? transferredBy : null,
                createdBy: user._id
            };
            const retVal = await spare_requests_and_transfers_model_1.default.create(isertData);
            return controller.sendSuccessResponse(res, {
                requestedData: retVal,
                message: `Item are successfully ${(0, lodash_1.capitalize)(type)}`,
            });
        }
        catch (error) {
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    validationErrors[key] = errors[key].message;
                });
                if (Object.keys(validationErrors).length > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors
                    });
                }
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while creating customer'
            });
        }
    }
    async assignTask(req, res) {
        try {
            const jobOrderId = req.params.id;
            if (!jobOrderId) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        jobOrderId: 'Job order master id is required'
                    }
                });
            }
            const validatedData = job_order_schema_1.assignTaskJobOrderSchema.safeParse(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
            let { taskId, orderDetailId, branchId, reassignedFromId, taskTitle, description, startDate, endDate, priority, taskStartTime, taskEndTime, assignmentHistory, technicianStatus } = validatedData.data;
            const user = res.locals.user;
            const orderDetails = await job_order_details_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(orderDetailId) });
            if ((0, lodash_1.isEmpty)(orderDetails)) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Job order not found!',
                });
            }
            if (orderDetails.serviceStatus === job_order_1.serviceStatusListJson.completed) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'This Job order already completed!',
                });
            }
            if (taskId && technicianStatus) { // check for technician status
                const taskDetails = await job_order_assignment_history_model_1.default.findOne({ jobOrderDetailId: new mongoose_1.default.Types.ObjectId(orderDetailId) });
                if ((0, lodash_1.isEmpty)(taskDetails)) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Your task is not found!',
                    });
                }
                if (taskDetails.technicianStatus === job_order_1.technicianStatusListJson.completed) {
                    return controller.sendErrorResponse(res, 200, {
                        message: `Already this task is ${(0, lodash_1.capitalize)(job_order_1.technicianStatusListJson.completed)}`,
                    });
                }
                // if (taskDetails.technicianStatus === technicianStatusListJson.canceled) {
                //     return controller.sendErrorResponse(res, 200, {
                //         message: `Already this task is ${capitalize(technicianStatusListJson.canceled)}`,
                //     });
                // }
            }
            const sequenceDoc = await sequence_model_1.default.findOneAndUpdate({ _id: 'taskSequence' }, { $inc: { sequenceValue: assignmentHistory.length } }, { new: true, upsert: true });
            if (sequenceDoc) {
                const startingTaskId = sequenceDoc.sequenceValue - assignmentHistory.length + 1;
                const bulkOperations = assignmentHistory.map((assignment, index) => {
                    return assignment?.taskId
                        ? {
                            updateOne: {
                                filter: { _id: new mongoose_1.default.Types.ObjectId(assignment?.taskId) },
                                update: {
                                    $set: {
                                        technicianId: assignment?.technicianId,
                                        taskTitle,
                                        description,
                                        startDate,
                                        endDate,
                                        priority,
                                        taskStartTime,
                                        taskEndTime,
                                        ...(reassignedFromId ? {
                                            reassignedFromId: new mongoose_1.default.Types.ObjectId(reassignedFromId),
                                            reassignedFromIdAt: new Date(),
                                        } : {}),
                                        ...(technicianStatus ? {
                                            technicianStatus,
                                            technicianStatusAt: new Date(),
                                        } : {}),
                                        assignedAt: new Date(),
                                    },
                                },
                            }
                        }
                        : {
                            insertOne: {
                                document: {
                                    jobMasterId: new mongoose_1.default.Types.ObjectId(jobOrderId),
                                    jobOrderDetailId: new mongoose_1.default.Types.ObjectId(orderDetailId),
                                    branchId: new mongoose_1.default.Types.ObjectId(branchId),
                                    technicianId: new mongoose_1.default.Types.ObjectId(assignment.technicianId),
                                    createdBy: user._id,
                                    taskId: startingTaskId + index,
                                    taskTitle,
                                    description,
                                    startDate,
                                    endDate,
                                    priority,
                                    taskStartTime,
                                    taskEndTime,
                                    ...(reassignedFromId ? {
                                        reassignedFromId: new mongoose_1.default.Types.ObjectId(reassignedFromId),
                                        reassignedFromIdAt: new Date(),
                                    } : {}),
                                    ...(technicianStatus ? {
                                        technicianStatus,
                                        technicianStatusAt: new Date(),
                                    } : {}),
                                    assignedAt: new Date(),
                                }
                            }
                        };
                });
                const retVal = await job_order_assignment_history_model_1.default.bulkWrite(bulkOperations, {
                    ordered: false,
                });
                if (retVal.insertedCount > 0 || retVal.matchedCount > 0 || retVal.upsertedCount > 0) {
                    if (taskId && technicianStatus && (technicianStatus === job_order_1.technicianStatusListJson.inProgress || technicianStatus === job_order_1.technicianStatusListJson.completed || technicianStatus === job_order_1.technicianStatusListJson.canceled)) {
                        const allTaskDetails = await job_order_assignment_history_model_1.default.find({ jobMasterId: orderDetails.jobMasterId });
                        if ((0, lodash_1.size)(allTaskDetails) > 0) {
                            const currentJobServiceTaskDetails = allTaskDetails.filter((task) => task.jobMasterId.toString() === orderDetails.jobMasterId.toString());
                            if ((0, lodash_1.size)(currentJobServiceTaskDetails) > 0) {
                                const allStatusesMatch = [
                                    job_order_1.technicianStatusListJson.inProgress,
                                    job_order_1.technicianStatusListJson.completed,
                                    job_order_1.technicianStatusListJson.canceled
                                ].find((status) => currentJobServiceTaskDetails.every((task) => task.technicianStatus === status));
                                if (!(0, lodash_1.isEmpty)(allStatusesMatch) && technicianStatus !== job_order_1.technicianStatusListJson.canceled) {
                                    await job_order_details_model_1.default.findByIdAndUpdate(orderDetails._id, {
                                        technicianStatus: allStatusesMatch,
                                        technicianStatusAt: new Date(),
                                        serviceStatus: allStatusesMatch,
                                        serviceStatusAt: new Date()
                                    }, { new: true, useFindAndModify: false });
                                    await job_order_master_model_1.default.findByIdAndUpdate(orderDetails.jobMasterId, {
                                        jobOrderStatus: allStatusesMatch,
                                        jobOrderStatusAt: new Date(),
                                    }, { new: true, useFindAndModify: false });
                                }
                                else {
                                    const checPartiallyCompleted = currentJobServiceTaskDetails.find((task) => task.technicianStatus === job_order_1.technicianStatusListJson.completed);
                                    if (!(0, lodash_1.isEmpty)(checPartiallyCompleted)) {
                                        await job_order_details_model_1.default.findByIdAndUpdate(orderDetails._id, {
                                            technicianStatus: job_order_1.technicianStatusListJson.completed,
                                            technicianStatusAt: new Date(),
                                            serviceStatus: job_order_1.serviceStatusListJson.completed,
                                            serviceStatusAt: new Date()
                                        }, { new: true, useFindAndModify: false });
                                        await job_order_master_model_1.default.findByIdAndUpdate(orderDetails.jobMasterId, {
                                            jobOrderStatus: job_order_1.jobOrderStatusListJson.partiallyCompleted,
                                            jobOrderStatusAt: new Date(),
                                        }, { new: true, useFindAndModify: false });
                                    }
                                }
                            }
                            else {
                            }
                        }
                        else {
                        }
                    }
                    const jobOrder = await job_order_service_1.default.findOneSimple({
                        jobDetailQuery: {
                            $or: [
                                { 'transferBranchId': orderDetails?.transferBranchId },
                                { 'branchId': orderDetails?.branchId },
                            ],
                        },
                        jobOrderId, simple: '0', jobdetails: '1', getcustomer: '1',
                    });
                    return controller.sendSuccessResponse(res, {
                        requestedData: jobOrder,
                        message: 'Assignment history updated/inserted successfully',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'No changes made to assignment history',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Failed to generate task IDs.',
                });
            }
        }
        catch (error) {
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    validationErrors[key] = errors[key].message;
                });
                if (Object.keys(validationErrors).length > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors
                    });
                }
            }
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async serviceStatusChange(req, res) {
        try {
            const orderMasterId = req.params.detailid;
            const orderMasterDetails = await job_order_master_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(orderMasterId) });
            if ((0, lodash_1.isEmpty)(orderMasterDetails)) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Job order not found!',
                });
            }
            const validatedData = job_order_schema_1.serviceStatusChangeJobOrderSchema.safeParse(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
            const { orderDetailId, serviceStatus, serviceStatusMessage } = validatedData.data;
            const orderDetails = await job_order_details_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(orderDetailId) });
            if ((0, lodash_1.isEmpty)(orderDetails)) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Job order not found!',
                });
            }
            if (orderDetails.serviceStatus === job_order_1.serviceStatusListJson.completed && (serviceStatus !== job_order_1.serviceStatusListJson.reopen && serviceStatus !== job_order_1.serviceStatusListJson.closed)) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Cannot change status, service is already completed!',
                });
            }
            if (serviceStatus === job_order_1.serviceStatusListJson.completed || serviceStatus === job_order_1.serviceStatusListJson.closed) {
                const allTaskDetails = await job_order_assignment_history_model_1.default.find({ jobMasterId: orderDetails.jobMasterId });
                const currentJobServiceTaskDetails = allTaskDetails.filter((task) => task.jobMasterId.toString() === orderDetails.jobMasterId.toString());
                if ((0, lodash_1.size)(currentJobServiceTaskDetails) > 0) {
                    const allStatusesMatch = [
                        job_order_1.technicianStatusListJson.assigned,
                        job_order_1.technicianStatusListJson.inProgress,
                    ].find((status) => currentJobServiceTaskDetails.every((task) => task.technicianStatus === status));
                    if ((0, lodash_1.isEmpty)(allStatusesMatch)) {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'Cannot change status, technician not completed this service!',
                        });
                    }
                }
            }
            let updateFields = { serviceStatus };
            const currentDate = new Date();
            if (serviceStatus === job_order_1.serviceStatusListJson.inProgress) {
                updateFields.serviceinProgressAt = currentDate;
            }
            else if (serviceStatus === job_order_1.serviceStatusListJson.completed) {
                updateFields.serviceCompletedAt = currentDate;
            }
            else if (serviceStatus === job_order_1.serviceStatusListJson.canceled) {
                updateFields.serviceCanceledAt = currentDate;
            }
            else if (serviceStatus === job_order_1.serviceStatusListJson.reopen) {
                updateFields.serviceReopenAt = currentDate;
            }
            else if (serviceStatus === job_order_1.serviceStatusListJson.closed) {
                updateFields.serviceClosedAt = currentDate;
            }
            const updatedJobDetailData = await job_order_details_model_1.default.findOneAndUpdate({ _id: orderDetails._id }, { $set: updateFields });
            const allOrderDetails = await job_order_details_model_1.default.find({ jobMasterId: orderMasterDetails._id });
            if ((0, lodash_1.isEmpty)(allOrderDetails) && (0, lodash_1.size)(allOrderDetails) === 0) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Job order not found!',
                });
            }
            const allStatuses = allOrderDetails.map((detail) => detail.serviceStatus);
            let newJobOrderStatus;
            if (allStatuses.every((status) => status === job_order_1.serviceStatusListJson.completed)) {
                newJobOrderStatus = job_order_1.serviceStatusListJson.completed;
            }
            else if (allStatuses.every((status) => status === job_order_1.serviceStatusListJson.canceled)) {
                newJobOrderStatus = job_order_1.serviceStatusListJson.canceled;
            }
            else if (allStatuses.some((status) => status === job_order_1.serviceStatusListJson.inProgress)) {
                newJobOrderStatus = job_order_1.serviceStatusListJson.inProgress;
            }
            else if (allStatuses.every((status) => status === job_order_1.serviceStatusListJson.reopen)) {
                newJobOrderStatus = job_order_1.serviceStatusListJson.reopen;
            }
            else if (allStatuses.every((status) => status === job_order_1.serviceStatusListJson.closed)) {
                newJobOrderStatus = job_order_1.serviceStatusListJson.closed;
            }
            else if (allStatuses.some((status) => status === job_order_1.serviceStatusListJson.canceled) && allStatuses.some((status) => status === job_order_1.serviceStatusListJson.completed)) {
                newJobOrderStatus = job_order_1.jobOrderStatusListJson.partiallyCompleted;
            }
            else if (allStatuses.some((status) => status === job_order_1.serviceStatusListJson.inProgress) && allStatuses.some((status) => status === job_order_1.serviceStatusListJson.completed)) {
                newJobOrderStatus = job_order_1.jobOrderStatusListJson.partiallyCompleted;
            }
            else {
                newJobOrderStatus = job_order_1.serviceStatusListJson.pending;
            }
            console.log('newJobOrderStatus', allStatuses);
            let jobOrderMaserDetail = null;
            if (orderMasterDetails.jobOrderStatus !== newJobOrderStatus) {
                jobOrderMaserDetail = await job_order_master_model_1.default.findOneAndUpdate({ _id: orderMasterDetails._id }, { $set: { jobOrderStatus: newJobOrderStatus } });
            }
            await job_order_customer_item_history_model_1.default.updateOne({
                jobOrderMasterId: orderMasterDetails._id,
                customerId: orderMasterDetails.customerId,
                jobOrderDetailId: orderDetailId,
                branchId: orderDetails.branchId,
                historyType: job_order_1.jobOrderCustomerItemTransferHistoryJson.created,
                serviceStatusMessage: serviceStatusMessage
            }, {
                $set: {
                    historyType: job_order_1.jobOrderCustomerItemTransferHistoryJson.created,
                    serviceStatusMessage: serviceStatusMessage,
                    historyStatus: newJobOrderStatus,
                }
            }, { upsert: true });
            const jobOrder = await job_order_service_1.default.findOneSimple({
                jobDetailQuery: {
                    $or: [
                        { 'transferBranchId': updatedJobDetailData?.transferBranchId },
                        { 'branchId': updatedJobDetailData?.branchId },
                    ],
                },
                jobOrderId: orderDetails.jobMasterId, simple: '0', jobdetails: '1', getcustomer: '1',
            });
            return controller.sendSuccessResponse(res, {
                requestedData: jobOrder,
                message: 'Service status scessfully completed'
            });
        }
        catch (error) {
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    validationErrors[key] = errors[key].message;
                });
                if (Object.keys(validationErrors).length > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors
                    });
                }
            }
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async assignVendors(req, res) {
        try {
            const orderDetailId = req.params.detailid;
            const orderDetails = await job_order_details_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(orderDetailId) });
            if ((0, lodash_1.isEmpty)(orderDetails)) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Job order not found!',
                });
            }
            const validatedData = job_order_schema_1.vendorAssignJobOrderSchema.safeParse(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
            const { vendorId, vendorNotes, vendorStatus, vendorAssignRemarks } = validatedData.data;
            const updatedJobDetailData = await job_order_details_model_1.default.findOneAndUpdate({ _id: orderDetails._id }, {
                $set: {
                    vendorId: new mongoose_1.default.Types.ObjectId(vendorId),
                    vendorAssignRemarks,
                    vendorNotes,
                    vendorStatus,
                    vendorStatusAt: new Date(),
                },
            });
            const jobOrder = await job_order_service_1.default.findOneSimple({
                jobDetailQuery: {
                    $or: [
                        { 'transferBranchId': updatedJobDetailData?.transferBranchId },
                        { 'branchId': updatedJobDetailData?.branchId },
                    ],
                },
                jobOrderId: orderDetails.jobMasterId, simple: '0', jobdetails: '1', getcustomer: '1',
            });
            return controller.sendSuccessResponse(res, {
                requestedData: jobOrder,
                message: 'Vendor Assigned scessfully completed'
            });
        }
        catch (error) {
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    validationErrors[key] = errors[key].message;
                });
                if (Object.keys(validationErrors).length > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors
                    });
                }
            }
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async serviceTransferBranch(req, res) {
        try {
            const orderDetailId = req.params.detailid;
            const orderDetails = await job_order_details_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(orderDetailId) });
            if ((0, lodash_1.isEmpty)(orderDetails)) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Job order not found!',
                });
            }
            const validatedData = job_order_schema_1.serviceTransferBranchJobOrderSchema.safeParse(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
            const { currentBranchId, transferBranchId, driverId, driverHistoryStatus, transferBranchStatus, serviceStatusMessage, transferReason } = validatedData.data;
            const updatedJobDetailData = await job_order_details_model_1.default.findOneAndUpdate({ _id: orderDetails._id }, {
                $set: {
                    ...(transferBranchId ? { branchId: transferBranchId, transferBranchId: currentBranchId } : {}),
                    transferBranchStatus,
                    transferBranchStatusAt: new Date(),
                    ...(driverId ? {
                        driverId,
                        driverHistoryStatus: job_order_1.customerItemTransferStatusJson.initiated === transferBranchStatus ? job_order_1.customerItemTransferDriverStatusJson.assigned : driverHistoryStatus,
                        driverHistoryStatusAt: new Date(),
                    } : {}),
                    ...((job_order_1.customerItemTransferStatusJson.initiated === transferBranchStatus || job_order_1.customerItemTransferStatusJson.rejected === transferBranchStatus) ? {
                        transferBranchStatus,
                        transferBranchStatusAt: new Date(),
                        transferBranchAt: new Date(),
                        transferReason,
                    } : {})
                },
            });
            const jobMaster = await job_order_master_model_1.default.findOne({ _id: orderDetails.jobMasterId }).select('jobId customerId').exec();
            const customerId = jobMaster?.customerId;
            const branchIds = [new mongoose_1.default.Types.ObjectId(currentBranchId)];
            if (transferBranchId) {
                branchIds.push(new mongoose_1.default.Types.ObjectId(transferBranchId));
            }
            const branchDetails = await branch_model_1.default.find({
                _id: { $in: branchIds }
            });
            const branchTitles = branchDetails.reduce((acc, branch) => {
                acc[branch._id.toString()] = branch.branchTitle;
                return acc;
            }, {});
            const currentBranchTitle = branchTitles[new mongoose_1.default.Types.ObjectId(currentBranchId).toString()];
            let transferBranchTitle = '';
            if (transferBranchId) {
                transferBranchTitle = branchTitles[new mongoose_1.default.Types.ObjectId(transferBranchId).toString()];
            }
            let driverMessage = '';
            if (!(0, lodash_1.isEmpty)(driverId)) {
                const driverDetail = await admin_user_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(driverId) });
                driverMessage = !(0, lodash_1.isEmpty)(driverDetail) ? ` with driver ${driverDetail.userTitle}` : '';
            }
            const transferNotes = transferBranchStatus === job_order_1.customerItemTransferStatusJson.received
                ? `The item received successfully at ${currentBranchTitle}.`
                : transferBranchStatus === job_order_1.customerItemTransferStatusJson.rejected
                    ? `The transfer was rejected from ${currentBranchTitle} for job order ${jobMaster?.jobId}.`
                    : (transferBranchId
                        ? `Transfer ${transferBranchStatus} from ${currentBranchTitle} to ${transferBranchTitle} for job order ${jobMaster?.jobId}${driverMessage}.`
                        : `Transfer ${transferBranchStatus} from ${currentBranchTitle} for job order ${jobMaster?.jobId}${driverMessage}.`);
            const updateData = {
                customerId,
                jobOrderMasterId: orderDetails.jobMasterId,
                jobOrderDetailId: new mongoose_1.default.Types.ObjectId(orderDetailId),
                branchId: transferBranchId ? transferBranchId : currentBranchId,
                transferFromBranchId: transferBranchId ? currentBranchId : updatedJobDetailData?.transferBranchId,
                historyType: job_order_1.jobOrderCustomerItemTransferHistoryJson.transfer,
                serviceStatusMessage: serviceStatusMessage,
                historyStatus: transferBranchStatus,
                ...(driverId ? {
                    driverId,
                    driverHistoryStatus: job_order_1.customerItemTransferStatusJson.initiated === transferBranchStatus ? job_order_1.customerItemTransferDriverStatusJson.assigned : driverHistoryStatus,
                    driverHistoryStatusAt: new Date(),
                } : {}),
                ...(transferReason !== '' ? { transferNotes: transferNotes + transferReason } : {})
            };
            await job_order_customer_item_history_model_1.default.updateOne({
                jobOrderMasterId: orderDetails.jobMasterId,
                jobOrderDetailId: new mongoose_1.default.Types.ObjectId(orderDetailId),
                branchId: transferBranchId ? transferBranchId : currentBranchId,
                transferFromBranchId: transferBranchId ? currentBranchId : updatedJobDetailData?.transferBranchId,
                historyType: job_order_1.jobOrderCustomerItemTransferHistoryJson.transfer,
                historyStatus: transferBranchStatus,
            }, { $set: updateData }, { upsert: true });
            const jobOrder = await job_order_service_1.default.findOneSimple({
                jobDetailQuery: {
                    $or: [
                        { 'transferBranchId': updatedJobDetailData?.transferBranchId },
                        { 'branchId': updatedJobDetailData?.branchId },
                    ],
                },
                jobOrderId: orderDetails.jobMasterId, simple: '0', jobdetails: '1', getcustomer: '1',
            });
            return controller.sendSuccessResponse(res, {
                requestedData: jobOrder,
                message: 'Service item transferred scessfully completed'
            });
        }
        catch (error) {
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    validationErrors[key] = errors[key].message;
                });
                if (Object.keys(validationErrors).length > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors
                    });
                }
            }
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async serviceTransferDriverAssign(req, res) {
        try {
            const orderDetailId = req.params.id;
            const orderDetails = await job_order_details_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(orderDetailId) });
            if ((0, lodash_1.isEmpty)(orderDetails)) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Job order not found!',
                });
            }
            const validatedData = job_order_schema_1.serviceTransferBranchJobOrderSchema.safeParse(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
            const { orderMasterId, currentBranchId, driverId, transferNotes, driverHistoryStatus } = validatedData.data;
            const serviceStatusMessage = (() => {
                if (driverHistoryStatus === job_order_1.customerItemTransferDriverStatusJson.assigned) {
                    return "The driver has been assigned to the job.";
                }
                else if (driverHistoryStatus === job_order_1.customerItemTransferDriverStatusJson.delivered) {
                    return "The items have been delivered by the driver.";
                }
                else {
                    return "Driver status is unknown.";
                }
            })();
            const updateData = {
                jobOrderMasterId: orderDetails.jobMasterId,
                jobOrderDetailId: new mongoose_1.default.Types.ObjectId(orderDetailId),
                branchId: new mongoose_1.default.Types.ObjectId(currentBranchId),
                driverId: new mongoose_1.default.Types.ObjectId(driverId),
                historyType: job_order_1.jobOrderCustomerItemTransferHistoryJson.driverTransfer,
                transferNotes,
                serviceStatusMessage,
                driverHistoryStatus,
                driverHistoryStatusAt: new Date(),
            };
            await job_order_customer_item_history_model_1.default.updateOne({
                jobOrderMasterId: orderDetails.jobMasterId,
                jobOrderDetailId: new mongoose_1.default.Types.ObjectId(orderDetailId),
                branchId: new mongoose_1.default.Types.ObjectId(currentBranchId),
                driverId: new mongoose_1.default.Types.ObjectId(driverId),
                historyType: job_order_1.jobOrderCustomerItemTransferHistoryJson.driverTransfer,
                driverHistoryStatus
            }, { $set: updateData }, { upsert: true });
            const updatedJobDetailData = await job_order_details_model_1.default.findOneAndUpdate({ _id: orderDetails._id }, {
                $set: {
                    driverId,
                    driverHistoryStatus,
                    driverHistoryStatusAt: new Date(),
                },
            });
            const jobOrder = await job_order_service_1.default.findOneSimple({
                jobDetailQuery: {
                    $or: [
                        { 'transferBranchId': updatedJobDetailData?.transferBranchId },
                        { 'branchId': updatedJobDetailData?.branchId },
                    ],
                },
                jobOrderId: orderDetails.jobMasterId, simple: '0', jobdetails: '1', getcustomer: '1',
            });
            return controller.sendSuccessResponse(res, {
                requestedData: jobOrder,
                message: 'Service item transferred scessfully completed'
            });
        }
        catch (error) {
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    validationErrors[key] = errors[key].message;
                });
                if (Object.keys(validationErrors).length > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors
                    });
                }
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while creating customer'
            });
        }
    }
    async findJobOrderAssignmentHistory(req, res) {
        try {
            const jobOrderId = req.params.id;
            const jobOrderDetailId = req.params.detailid;
            const assignmentHistories = await job_order_assignment_history_model_1.default.find({
                jobMasterId: new mongoose_1.default.Types.ObjectId(jobOrderId),
                ...(jobOrderDetailId !== '' ? { jobOrderDetailId: new mongoose_1.default.Types.ObjectId(jobOrderDetailId) } : {})
            });
            return controller.sendSuccessResponse(res, {
                requestedData: assignmentHistories,
                message: 'Assignment history ',
            });
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async findOneSimple(req, res) {
        try {
            const jobOrderId = req.params.id;
            if (jobOrderId) {
                const { simple = '1', jobdetails = '0', getcustomer = '0', getbranch = '0' } = req.query;
                const user = res.locals.user;
                const visitorBranchId = (0, project_utils_1.getVisitorBranchId)(user);
                const contextTechnicianId = (0, project_utils_1.getTechnicianId)(user);
                let jobDetailQuery = { _id: { $exists: true } };
                if (visitorBranchId) {
                    jobDetailQuery.$or = [
                        { 'branchId': new mongoose_1.default.Types.ObjectId(visitorBranchId) },
                        { 'transferBranchId': new mongoose_1.default.Types.ObjectId(visitorBranchId) },
                    ];
                }
                const jobOrder = await job_order_service_1.default.findOneSimple({ jobOrderId, simple, jobdetails, getcustomer, getbranch, contextTechnicianId, jobDetailQuery });
                return controller.sendSuccessResponse(res, {
                    requestedData: jobOrder,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'JobOrder Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = job_order_schema_1.createJobOrderSchema.safeParse(req.body);
            if (validatedData.success) {
                const jobOrderId = req.params.id;
                if (jobOrderId) {
                    const updatedJobOrderData = req.body;
                    const updatedJobOrder = await job_order_service_1.default.update(jobOrderId, { ...updatedJobOrderData, updatedAt: new Date() });
                    if (updatedJobOrder) {
                        return controller.sendSuccessResponse(res, {
                            requestedData: updatedJobOrder,
                            message: 'JobOrder updated successfully!'
                        }, 200, {
                            sourceFromId: updatedJobOrder._id,
                            sourceFrom: task_log_1.adminTaskLog.jobOrder.jobOrder,
                            activity: task_log_1.adminTaskLogActivity.update,
                            activityStatus: task_log_1.adminTaskLogStatus.success,
                            activityComment: "JobOrder updated successfully"
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'JobOrder Id not found!',
                        });
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'JobOrder Id not found! Please try again with JobOrder id',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating jobOrder'
            });
        }
    }
    async destroy(req, res) {
        try {
            const jobOrderId = req.params.id;
            if (jobOrderId) {
                const jobOrder = await job_order_service_1.default.findOneSimple(jobOrderId);
                if (jobOrder) {
                    // await JobOrderService.destroy(jobOrderId);
                    // return controller.sendSuccessResponse(res, { message: 'JobOrder deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this jobOrder',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This jobOrder details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'JobOrder id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting jobOrder' });
        }
    }
}
exports.default = new JobOrderController();
