"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../base-controller"));
const branch_service_1 = __importDefault(require("../../../services/admin/operations/branch-service"));
const branch_schema_1 = require("../../../utils/schemas/admin/operation/branch-schema");
const mongoose_1 = __importDefault(require("mongoose"));
const controller = new base_controller_1.default();
class BranchController extends base_controller_1.default {
    async findAll(req, res) {
        const { branchId, branchCode, itemId = '', getItemStock = '0', currentBranchId, page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (branchId) {
            query._id = new mongoose_1.default.Types.ObjectId(itemId);
        }
        if (branchCode) {
            query.branchCode = new RegExp(branchCode, 'i');
        }
        if (itemId) {
            query = {
                ...query,
                'stockInventory.itemId': new mongoose_1.default.Types.ObjectId('66e82e0306cadf5f7a44a15b'),
                'stockInventory.itemStock': { $gt: 0 }
            };
        }
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { branchTitle: keywordRegex }
                ],
                ...query
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const branch = await branch_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                getItemStock,
                currentBranchId,
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: branch.paginatedResults,
                totalCount: branch.totalCount,
                ...(currentBranchId ? { currentBranch: branch.currentBranch } : {}),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching branch' });
        }
    }
    async findAllBranchVisitors(req, res) {
        const { branchId, visitorId, page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (branchId) {
            query.branchId = new mongoose_1.default.Types.ObjectId(branchId);
        }
        if (visitorId) {
            query.visitorId = visitorId;
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const branch = await branch_service_1.default.findAllBranchVisitors({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: branch,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching branch' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = branch_schema_1.branchchema.safeParse(req.body);
            if (validatedData.success) {
                const { branchTitle, branchCode, slug } = validatedData.data;
                const user = res.locals.user;
                const branchData = {
                    branchTitle,
                    slug: (0, helpers_1.slugifyManually)(branchTitle),
                    branchCode,
                    status: '1',
                    createdBy: user._id,
                    createdAt: new Date()
                };
                const newBranch = await branch_service_1.default.create(branchData);
                return controller.sendSuccessResponse(res, {
                    requestedData: newBranch,
                    message: 'Branch created successfully!'
                }, 200, {
                    sourceFromId: newBranch._id,
                    sourceFrom: task_log_1.adminTaskLog.operation.branch,
                    activity: task_log_1.adminTaskLogActivity.create,
                    activityStatus: task_log_1.adminTaskLogStatus.success
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            if (error && error.errors && error.errors.branchTitle && error.errors.branchTitle.properties) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        branchTitle: error.errors.branchTitle.properties.message
                    }
                });
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while creating branch',
            });
        }
    }
    async findOne(req, res) {
        try {
            const branchId = req.params.id;
            if (branchId) {
                const branch = await branch_service_1.default.findOne(branchId);
                return controller.sendSuccessResponse(res, {
                    requestedData: branch,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Branch Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = branch_schema_1.branchchema.safeParse(req.body);
            if (validatedData.success) {
                const branchId = req.params.id;
                if (branchId) {
                    const updatedBranchData = req.body;
                    const updatedBranch = await branch_service_1.default.update(branchId, { ...updatedBranchData, updatedAt: new Date() });
                    if (updatedBranch) {
                        return controller.sendSuccessResponse(res, {
                            requestedData: updatedBranch,
                            message: 'Branch updated successfully!'
                        }, 200, {
                            sourceFromId: updatedBranch._id,
                            sourceFrom: task_log_1.adminTaskLog.operation.branch,
                            activity: task_log_1.adminTaskLogActivity.update,
                            activityStatus: task_log_1.adminTaskLogStatus.success
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'Branch Id not found!',
                        });
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Branch Id not found! Please try again with Branch id',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating branch'
            });
        }
    }
    async destroy(req, res) {
        try {
            const branchId = req.params.id;
            if (branchId) {
                const branch = await branch_service_1.default.findOne(branchId);
                if (branch) {
                    // await BranchService.destroy(branchId);
                    // return controller.sendSuccessResponse(res, { message: 'Branch deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this branch',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This branch details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Branch id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting branch' });
        }
    }
}
exports.default = new BranchController();
