"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../base-controller"));
const warehouse_service_1 = __importDefault(require("../../../services/admin/operations/warehouse-service"));
const warehouse_schema_1 = require("../../../utils/schemas/admin/operation/warehouse-schema");
const branch_model_1 = __importDefault(require("../../../model/admin/operations/branch-model"));
const controller = new base_controller_1.default();
class WarehouseController extends base_controller_1.default {
    async findAll(req, res) {
        const { branchId, page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (branchId) {
            query._id = new mongoose_1.default.Types.ObjectId(branchId);
        }
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { warehouseTitle: keywordRegex }
                ],
                ...query
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const warehouse = await warehouse_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: warehouse,
                totalCount: await warehouse_service_1.default.getTotalCount(query),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching warehouse' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = warehouse_schema_1.warehousechema.safeParse(req.body);
            if (validatedData.success) {
                const { branchCode, warehouseTitle, warehouseCode, slug } = validatedData.data;
                const user = res.locals.user;
                const branchDetails = await branch_model_1.default.findOne({ branchCode }).select('_id branchCode').lean();
                if (!branchDetails) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'The branch code is not found',
                    });
                }
                const warehouseData = {
                    branchId: branchDetails._id,
                    warehouseTitle,
                    slug: (0, helpers_1.slugifyManually)(warehouseTitle),
                    warehouseCode,
                    status: '1',
                    createdBy: user._id,
                    createdAt: new Date()
                };
                const newWarehouse = await warehouse_service_1.default.create(warehouseData);
                return controller.sendSuccessResponse(res, {
                    requestedData: newWarehouse,
                    message: 'Warehouse created successfully!'
                }, 200, {
                    sourceFromId: newWarehouse._id,
                    sourceFrom: task_log_1.adminTaskLog.operation.warehouse,
                    activity: task_log_1.adminTaskLogActivity.create,
                    activityStatus: task_log_1.adminTaskLogStatus.success
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            if (error && error.errors && error.errors.warehouseTitle && error.errors.warehouseTitle.properties) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        warehouseTitle: error.errors.warehouseTitle.properties.message
                    }
                });
            }
            else if (error && error.errors && error.errors.branchCode && error.errors.branchCode.properties) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        branchCode: error.errors.branchCode.properties.message
                    }
                });
            }
            else if (error && error.errors && error.errors.slug && error.errors.slug.properties) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        slug: error.errors.slug.properties.message
                    }
                });
            }
            else if (error && error.errors && error.errors.warehouseCode && error.errors.warehouseCode.properties) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        warehouseCode: error.errors.warehouseCode.properties.message
                    }
                });
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while creating warehouse',
            });
        }
    }
    async findOne(req, res) {
        try {
            const warehouseId = req.params.id;
            if (warehouseId) {
                const warehouse = await warehouse_service_1.default.findOne(warehouseId);
                return controller.sendSuccessResponse(res, {
                    requestedData: warehouse,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Warehouse Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = warehouse_schema_1.warehousechema.safeParse(req.body);
            if (validatedData.success) {
                const warehouseId = req.params.id;
                if (warehouseId) {
                    const updatedWarehouseData = req.body;
                    const updatedWarehouse = await warehouse_service_1.default.update(warehouseId, { ...updatedWarehouseData, updatedAt: new Date() });
                    if (updatedWarehouse) {
                        return controller.sendSuccessResponse(res, {
                            requestedData: updatedWarehouse,
                            message: 'Warehouse updated successfully!'
                        }, 200, {
                            sourceFromId: updatedWarehouse._id,
                            sourceFrom: task_log_1.adminTaskLog.operation.warehouse,
                            activity: task_log_1.adminTaskLogActivity.update,
                            activityStatus: task_log_1.adminTaskLogStatus.success
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'Warehouse Id not found!',
                        });
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Warehouse Id not found! Please try again with Warehouse id',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating warehouse'
            });
        }
    }
    async destroy(req, res) {
        try {
            const warehouseId = req.params.id;
            if (warehouseId) {
                const warehouse = await warehouse_service_1.default.findOne(warehouseId);
                if (warehouse) {
                    // await WarehouseService.destroy(warehouseId);
                    // return controller.sendSuccessResponse(res, { message: 'Warehouse deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this warehouse',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This warehouse details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Warehouse id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting warehouse' });
        }
    }
}
exports.default = new WarehouseController();
