"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../base-controller"));
const zod_1 = require("zod");
const additional_cost_values_service_1 = __importDefault(require("../../../services/other/additional-cost-values-service"));
const additional_cost_values_schema_1 = require("../../../utils/schemas/admin/other/additional-cost-values-schema");
const controller = new base_controller_1.default();
class AdditionalCostValuesController extends base_controller_1.default {
    async findAll(req, res) {
        const { page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { costValueTitle: keywordRegex }
                ],
                ...query
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const additionalCostValues = await additional_cost_values_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: additionalCostValues.data,
                totalCount: additionalCostValues.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching Additional cost values' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = await additional_cost_values_schema_1.additionalCostValuesSchema.parseAsync(req.body);
            const { costValueTitle, slug, operationType, defaultValue, status } = validatedData;
            const user = res.locals.user;
            const additionalCostValuesData = {
                costValueTitle,
                operationType,
                slug: (0, helpers_1.slugify)(costValueTitle),
                defaultValue,
                status: status || '1',
                createdBy: user._id,
                createdAt: new Date(),
                updatedAt: new Date(),
            };
            const newAdditionalCostValues = await additional_cost_values_service_1.default.create(additionalCostValuesData);
            return controller.sendSuccessResponse(res, {
                requestedData: newAdditionalCostValues,
                message: 'Additional cost values created successfully!',
            }, 200, {
                sourceFromId: newAdditionalCostValues?._id,
                sourceFrom: task_log_1.adminTaskLog.other.additionalCostValues,
                activity: task_log_1.adminTaskLogActivity.create,
                activityStatus: task_log_1.adminTaskLogStatus.success,
                activityComment: "Additional Cost Values created successfully"
            });
        }
        catch (error) {
            if (error instanceof zod_1.ZodError) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(error.errors),
                });
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'An error occurred while creating the additionalCostValues.',
            });
        }
    }
    async findOne(req, res) {
        try {
            const additionalCostValuesId = req.params.id;
            if (additionalCostValuesId) {
                const additionalCostValues = await additional_cost_values_service_1.default.findOne(additionalCostValuesId);
                return controller.sendSuccessResponse(res, {
                    requestedData: additionalCostValues,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Additional cost values Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = await additional_cost_values_schema_1.additionalCostValuesSchema.safeParseAsync(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 400, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors),
                });
            }
            const additionalCostValuesId = req.params.id;
            if (additionalCostValuesId) {
                const updatedAdditionalCostValuesData = req.body;
                const updatedAdditionalCostValues = await additional_cost_values_service_1.default.update(additionalCostValuesId, { ...updatedAdditionalCostValuesData, updatedAt: new Date() });
                if (updatedAdditionalCostValues) {
                    return controller.sendSuccessResponse(res, {
                        requestedData: updatedAdditionalCostValues,
                        message: 'Additional cost values updated successfully!'
                    }, 200, {
                        sourceFromId: updatedAdditionalCostValues._id,
                        sourceFrom: task_log_1.adminTaskLog.other.additionalCostValues,
                        activity: task_log_1.adminTaskLogActivity.update,
                        activityStatus: task_log_1.adminTaskLogStatus.success,
                        activityComment: "Additional cost values updated successfully"
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Additional cost values Id not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Additional cost values Id not found! Please try again with Additional cost values id',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating Additional cost values'
            });
        }
    }
    async destroy(req, res) {
        try {
            const additionalCostValuesId = req.params.id;
            if (additionalCostValuesId) {
                const additionalCostValues = await additional_cost_values_service_1.default.findOne(additionalCostValuesId);
                if (additionalCostValues) {
                    // await AdditionalCostValuesService.destroy(additionalCostValuesId);
                    // return controller.sendSuccessResponse(res, { message: 'AdditionalCostValues deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this additionalCostValues',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This additionalCostValues details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'AdditionalCostValues id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting additionalCostValues' });
        }
    }
}
exports.default = new AdditionalCostValuesController();
