"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../base-controller"));
const zod_1 = require("zod");
const category_model_1 = __importDefault(require("../../../model/admin/inventory-management/category-model"));
const conditions_service_1 = __importDefault(require("../../../services/other/conditions-service"));
const condition_schema_1 = require("../../../utils/schemas/admin/other/condition-schema");
const conditions_model_1 = __importDefault(require("../../../model/admin/other/conditions-model"));
const complaint_model_1 = __importDefault(require("../../../model/admin/inventory-management/complaint-model"));
const category_config_1 = require("../../../utils/config/admin/inventory-management/category-config");
const category_1 = require("../../../utils/admin/category");
const controller = new base_controller_1.default();
class ConditionController extends base_controller_1.default {
    async findAllComplaintsAndConditions(req, res) {
        const { categoryId, page_size = 1, limit = 10, status = ['0', '1', '2'], } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        let categoryIds = [];
        if (categoryId) {
            // async function fetchAllCategories(categoryIds: any[]): Promise<any[]> {
            //     let queue = [...categoryIds];
            //     const allCategoryIds = new Set([...categoryIds]);
            //     while (queue.length > 0) {
            //         const categoriesData = await CategoryModel.find(
            //             { parentCategory: { $in: queue } },
            //             '_id'
            //         );
            //         const childCategoryIds = categoriesData.map(category => category._id);
            //         if (childCategoryIds.length === 0) {
            //             break;
            //         }
            //         queue = childCategoryIds;
            //         childCategoryIds.forEach(id => allCategoryIds.add(id));
            //     }
            //     return Array.from(allCategoryIds);
            // }
            // const categoryIds = await fetchAllCategories([...new Set([new mongoose.Types.ObjectId(categoryId)])]);
            let categoryBatchIds = [];
            if (categoryId) {
                categoryBatchIds.push(new mongoose_1.default.Types.ObjectId(categoryId));
            }
            const uniqueCategoryIds = [...new Set(categoryBatchIds)];
            var categoriesWithChilleds = await category_model_1.default.aggregate((0, category_config_1.categoriesWithChildrenInArrayAggregation)(uniqueCategoryIds));
            categoryIds = (0, category_1.extractAllCategoryIds)(categoriesWithChilleds);
        }
        try {
            const conditions = await conditions_model_1.default.find({ categoryId: { $in: categoryIds } });
            const complaints = await complaint_model_1.default.find({ categoryId: { $in: categoryIds } });
            return controller.sendSuccessResponse(res, {
                requestedData: { conditions, complaints },
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching condition' });
        }
    }
    async findAll(req, res) {
        const { categoryId, page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { conditionTitle: keywordRegex }
                ],
                ...query
            };
        }
        if (categoryId) {
            async function fetchAllCategories(categoryIds) {
                let queue = [...categoryIds];
                const allCategoryIds = new Set([...categoryIds]);
                while (queue.length > 0) {
                    const categoriesData = await category_model_1.default.find({ parentCategory: { $in: queue } }, '_id');
                    const childCategoryIds = categoriesData.map(category => category._id);
                    if (childCategoryIds.length === 0) {
                        break;
                    }
                    queue = childCategoryIds;
                    childCategoryIds.forEach(id => allCategoryIds.add(id));
                }
                return Array.from(allCategoryIds);
            }
            const categoryIds = await fetchAllCategories([...new Set([new mongoose_1.default.Types.ObjectId(categoryId)])]);
            query = {
                ...query, categoryId: { $in: categoryIds }
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const condition = await conditions_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: condition.data,
                totalCount: condition.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching condition' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = await condition_schema_1.conditionSchema.parseAsync(req.body);
            const { categoryId, conditionTitle, slug, description, status } = validatedData;
            const user = res.locals.user;
            const conditionData = {
                categoryId,
                conditionTitle,
                description,
                slug: (0, helpers_1.slugify)(conditionTitle),
                status: status || 'active',
                createdBy: user._id,
                createdAt: new Date(),
                updatedAt: new Date(),
            };
            console.log(conditionData);
            const newCondition = await conditions_service_1.default.create(conditionData);
            return controller.sendSuccessResponse(res, {
                requestedData: newCondition,
                message: 'Condition created successfully!',
            }, 200, {
                sourceFromId: newCondition?._id,
                sourceFrom: task_log_1.adminTaskLog.other.conditions,
                activity: task_log_1.adminTaskLogActivity.create,
                activityStatus: task_log_1.adminTaskLogStatus.success,
                activityComment: "Condition created successfully"
            });
        }
        catch (error) {
            if (error instanceof zod_1.ZodError) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(error.errors),
                });
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'An error occurred while creating the condition.',
            });
        }
    }
    async findOne(req, res) {
        try {
            const conditionId = req.params.id;
            if (conditionId) {
                const condition = await conditions_service_1.default.findOne(conditionId);
                return controller.sendSuccessResponse(res, {
                    requestedData: condition,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Condition Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = await condition_schema_1.conditionSchema.safeParseAsync(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 400, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors),
                });
            }
            const conditionId = req.params.id;
            if (conditionId) {
                const updatedConditionData = req.body;
                const updatedCondition = await conditions_service_1.default.update(conditionId, { ...updatedConditionData, updatedAt: new Date() });
                if (updatedCondition) {
                    return controller.sendSuccessResponse(res, {
                        requestedData: updatedCondition,
                        message: 'Condition updated successfully!'
                    }, 200, {
                        sourceFromId: updatedCondition._id,
                        sourceFrom: task_log_1.adminTaskLog.other.conditions,
                        activity: task_log_1.adminTaskLogActivity.update,
                        activityStatus: task_log_1.adminTaskLogStatus.success,
                        activityComment: "Condition updated successfully"
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Condition Id not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Condition Id not found! Please try again with Condition id',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating condition'
            });
        }
    }
    async destroy(req, res) {
        try {
            const conditionId = req.params.id;
            if (conditionId) {
                const condition = await conditions_service_1.default.findOne(conditionId);
                if (condition) {
                    // await ConditionService.destroy(conditionId);
                    // return controller.sendSuccessResponse(res, { message: 'Condition deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this condition',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This condition details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Condition id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting condition' });
        }
    }
}
exports.default = new ConditionController();
