"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const zod_1 = require("zod");
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../base-controller"));
const category_1 = require("../../../utils/admin/category");
const category_model_1 = __importDefault(require("../../../model/admin/inventory-management/category-model"));
const customer_accessory_schema_1 = require("../../../utils/schemas/admin/other/customer-accessory.schema");
const customer_accessories_service_1 = __importDefault(require("../../../services/other/customer-accessories.service"));
const category_config_1 = require("../../../utils/config/admin/inventory-management/category-config");
const controller = new base_controller_1.default();
class CustomerAccessoriesController extends base_controller_1.default {
    async findAll(req, res) {
        const { categoryId, page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { customerAccessoryTitle: keywordRegex }
                ],
                ...query
            };
        }
        if (categoryId) {
            let categoryBatchIds = [];
            if (categoryId) {
                categoryBatchIds.push(new mongoose_1.default.Types.ObjectId(categoryId));
            }
            const uniqueCategoryIds = [...new Set(categoryBatchIds)];
            var categoriesWithChilleds = await category_model_1.default.aggregate((0, category_config_1.categoriesWithChildrenInArrayAggregation)(uniqueCategoryIds));
            const categoryIds = (0, category_1.extractAllCategoryIds)(categoriesWithChilleds);
            query = {
                ...query, categoryId: { $in: categoryIds }
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const customerAccessory = await customer_accessories_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: customerAccessory.data,
                totalCount: customerAccessory.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching customer accessory' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = await customer_accessory_schema_1.customerAccessorySchema.parseAsync(req.body);
            const { categoryId, customerAccessoryTitle, slug, description, status } = validatedData;
            const user = res.locals.user;
            const customerAccessoryData = {
                categoryId,
                customerAccessoryTitle,
                description,
                slug: (0, helpers_1.slugify)(customerAccessoryTitle),
                status: status || 'active',
                createdBy: user._id,
                createdAt: new Date(),
                updatedAt: new Date(),
            };
            console.log(customerAccessoryData);
            const newcustomerAccessory = await customer_accessories_service_1.default.create(customerAccessoryData);
            return controller.sendSuccessResponse(res, {
                requestedData: newcustomerAccessory,
                message: 'customer accessory created successfully!',
            }, 200, {
                sourceFromId: newcustomerAccessory?._id,
                sourceFrom: task_log_1.adminTaskLog.other.customerAccessories,
                activity: task_log_1.adminTaskLogActivity.create,
                activityStatus: task_log_1.adminTaskLogStatus.success,
                activityComment: "customer accessory created successfully"
            });
        }
        catch (error) {
            if (error instanceof zod_1.ZodError) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(error.errors),
                });
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'An error occurred while creating the customer accessory.',
            });
        }
    }
    async findOne(req, res) {
        try {
            const customerAccessoryId = req.params.id;
            if (customerAccessoryId) {
                const customerAccessory = await customer_accessories_service_1.default.findOne(customerAccessoryId);
                return controller.sendSuccessResponse(res, {
                    requestedData: customerAccessory,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'customer accessory Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = await customer_accessory_schema_1.customerAccessorySchema.safeParseAsync(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 400, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors),
                });
            }
            const customerAccessoryId = req.params.id;
            if (customerAccessoryId) {
                const updatedcustomerAccessoryData = req.body;
                const updatedcustomerAccessory = await customer_accessories_service_1.default.update(customerAccessoryId, { ...updatedcustomerAccessoryData, updatedAt: new Date() });
                if (updatedcustomerAccessory) {
                    return controller.sendSuccessResponse(res, {
                        requestedData: updatedcustomerAccessory,
                        message: 'customer accessory updated successfully!'
                    }, 200, {
                        sourceFromId: updatedcustomerAccessory._id,
                        sourceFrom: task_log_1.adminTaskLog.other.customerAccessories,
                        activity: task_log_1.adminTaskLogActivity.update,
                        activityStatus: task_log_1.adminTaskLogStatus.success,
                        activityComment: "customer accessory updated successfully"
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'customer accessory Id not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'customer accessory Id not found! Please try again with customer accessory id',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating customer accessory'
            });
        }
    }
    async destroy(req, res) {
        try {
            const customerAccessoryId = req.params.id;
            if (customerAccessoryId) {
                const customerAccessory = await customer_accessories_service_1.default.findOne(customerAccessoryId);
                if (customerAccessory) {
                    // await CustomerAccessoriesService.destroy(customerAccessoryId);
                    // return controller.sendSuccessResponse(res, { message: 'customerAccessory deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this customer accessory',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This customer accessory details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'customer accessory id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting customer accessory' });
        }
    }
}
exports.default = new CustomerAccessoriesController();
