"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../base-controller"));
const zod_1 = require("zod");
const category_model_1 = __importDefault(require("../../../model/admin/inventory-management/category-model"));
const service_status_service_1 = __importDefault(require("../../../services/other/service-status-service"));
const service_status_schema_1 = require("../../../utils/schemas/admin/other/service-status-schema");
const controller = new base_controller_1.default();
class ServiceStatusController extends base_controller_1.default {
    async findAll(req, res) {
        const { categoryId, parentCategoryId = '', page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { serviceStatusTitle: keywordRegex }
                ],
                ...query
            };
        }
        if (categoryId) {
            async function fetchAllCategories(categoryIds) {
                let queue = [...categoryIds];
                const allCategoryIds = new Set([...categoryIds]);
                while (queue.length > 0) {
                    const categoriesData = await category_model_1.default.find({ parentCategory: { $in: queue } }, '_id');
                    const childCategoryIds = categoriesData.map(category => category._id);
                    if (childCategoryIds.length === 0) {
                        break;
                    }
                    queue = childCategoryIds;
                    childCategoryIds.forEach(id => allCategoryIds.add(id));
                }
                return Array.from(allCategoryIds);
            }
            const categoryIds = await fetchAllCategories([...new Set([new mongoose_1.default.Types.ObjectId(categoryId)])]);
            query = {
                ...query, categoryId: { $in: categoryIds }
            };
        }
        if (parentCategoryId) {
            query = {
                ...query, categoryId: new mongoose_1.default.Types.ObjectId(parentCategoryId)
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const serviceStatus = await service_status_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: serviceStatus.data,
                totalCount: serviceStatus.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching serviceStatus' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = await service_status_schema_1.serviceStatusSchema.parseAsync(req.body);
            const { categoryId, serviceStatusTitle, slug, description, status } = validatedData;
            const user = res.locals.user;
            const serviceStatusData = {
                categoryId,
                serviceStatusTitle,
                description,
                slug: (0, helpers_1.slugify)(serviceStatusTitle),
                status: status || 'active',
                createdBy: user._id,
                createdAt: new Date(),
                updatedAt: new Date(),
            };
            const newServiceStatus = await service_status_service_1.default.create(serviceStatusData);
            return controller.sendSuccessResponse(res, {
                requestedData: newServiceStatus,
                message: 'ServiceStatus created successfully!',
            }, 200, {
                sourceFromId: newServiceStatus?._id,
                sourceFrom: task_log_1.adminTaskLog.other.serviceStatus,
                activity: task_log_1.adminTaskLogActivity.create,
                activityStatus: task_log_1.adminTaskLogStatus.success,
                activityComment: "ServiceStatus created successfully"
            });
        }
        catch (error) {
            if (error instanceof zod_1.ZodError) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(error.errors),
                });
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'An error occurred while creating the serviceStatus.',
            });
        }
    }
    async findOne(req, res) {
        try {
            const serviceStatusId = req.params.id;
            if (serviceStatusId) {
                const serviceStatus = await service_status_service_1.default.findOne(serviceStatusId);
                return controller.sendSuccessResponse(res, {
                    requestedData: serviceStatus,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'ServiceStatus Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = await service_status_schema_1.serviceStatusSchema.safeParseAsync(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 400, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors),
                });
            }
            const serviceStatusId = req.params.id;
            if (serviceStatusId) {
                const updatedServiceStatusData = req.body;
                const updatedServiceStatus = await service_status_service_1.default.update(serviceStatusId, { ...updatedServiceStatusData, updatedAt: new Date() });
                if (updatedServiceStatus) {
                    return controller.sendSuccessResponse(res, {
                        requestedData: updatedServiceStatus,
                        message: 'ServiceStatus updated successfully!'
                    }, 200, {
                        sourceFromId: updatedServiceStatus._id,
                        sourceFrom: task_log_1.adminTaskLog.other.serviceStatus,
                        activity: task_log_1.adminTaskLogActivity.update,
                        activityStatus: task_log_1.adminTaskLogStatus.success,
                        activityComment: "ServiceStatus updated successfully"
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'ServiceStatus Id not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'ServiceStatus Id not found! Please try again with ServiceStatus id',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating serviceStatus'
            });
        }
    }
    async destroy(req, res) {
        try {
            const serviceStatusId = req.params.id;
            if (serviceStatusId) {
                const serviceStatus = await service_status_service_1.default.findOne(serviceStatusId);
                if (serviceStatus) {
                    // await ServiceStatusService.destroy(serviceStatusId);
                    // return controller.sendSuccessResponse(res, { message: 'ServiceStatus deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this serviceStatus',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This serviceStatus details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'ServiceStatus id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting serviceStatus' });
        }
    }
}
exports.default = new ServiceStatusController();
