"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../../../controllers/admin/base-controller"));
const category_model_1 = __importDefault(require("../../../model/admin/service-management/category-model"));
const category_service_1 = __importDefault(require("../../../services/admin/service-management/category-service"));
const category_schema_1 = require("../../../utils/schemas/admin/service-management/category-schema");
const controller = new base_controller_1.default();
class CategoryController extends base_controller_1.default {
    async findAll(req, res) {
        try {
            const { page_size = 1, limit = '', status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '', slug = '', category = '', categoryId = '', _id = '', parentCategory = '' } = req.query;
            let query = { _id: { $exists: true } };
            let categoryIdCheck;
            if (status && status !== '') {
                query.status = { $in: Array.isArray(status) ? status : [status] };
            }
            else {
                query.status = '1';
            }
            if (keyword) {
                const keywordRegex = new RegExp(keyword, 'i');
                query = {
                    $or: [
                        { categoryTitle: keywordRegex },
                        { slug: keywordRegex },
                    ],
                    ...query
                };
            }
            if (categoryId) {
                query = {
                    ...query, _id: new mongoose_1.default.Types.ObjectId(categoryId)
                };
            }
            if (_id) {
                if (typeof _id === 'string') {
                    query = {
                        ...query, _id: new mongoose_1.default.Types.ObjectId(_id)
                    };
                }
                else {
                    const categoryIds = _id.map((id) => new mongoose_1.default.Types.ObjectId(id));
                    categoryIdCheck = {
                        _id: { $in: categoryIds }
                    };
                }
            }
            if (categoryIdCheck && (Object.keys(categoryIdCheck)).length > 0) {
                query = {
                    ...query, ...categoryIdCheck
                };
            }
            if (category) {
                query = {
                    ...query, _id: new mongoose_1.default.Types.ObjectId(category)
                };
            }
            if (slug) {
                query = {
                    ...query, slug: slug
                };
            }
            if (parentCategory) {
                query = {
                    ...query, parentCategory: new mongoose_1.default.Types.ObjectId(parentCategory)
                };
            }
            const sort = {};
            if (sortby && sortorder) {
                sort[sortby] = sortorder === 'desc' ? -1 : 1;
            }
            const categories = await category_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            controller.sendSuccessResponse(res, {
                requestedData: categories,
                totalCount: await category_service_1.default.getTotalCount(query),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching categories' });
        }
    }
    async findAllChilledCategories(req, res) {
        try {
            const { status = '1', keyword = '', categoryId = '', parentCategory = '' } = req.query;
            let query = { _id: { $exists: true } };
            if (status && status !== '') {
                query.status = { $in: Array.isArray(status) ? status : [status] };
            }
            else {
                query.status = '1';
            }
            if (keyword) {
                const keywordRegex = new RegExp(keyword, 'i');
                query = {
                    $or: [
                        { categoryTitle: keywordRegex },
                        { slug: keywordRegex },
                    ],
                    ...query
                };
            }
            if (categoryId) {
                query = { _id: categoryId };
            }
            if (!parentCategory && keyword === '' && categoryId === '') {
                query.$or = [
                    { parentCategory: null },
                    { parentCategory: { $exists: false } }
                ];
            }
            else if (parentCategory) {
                query.parentCategory = parentCategory;
            }
            const categories = await category_service_1.default.findAllChilledCategories(query);
            controller.sendSuccessResponse(res, {
                requestedData: categories,
                totalCount: await category_service_1.default.getTotalCount(query),
                message: 'Success!ddd'
            }, 200);
        }
        catch (error) {
            controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching categories' });
        }
    }
    async findAllParentCategories(req, res) {
        try {
            const { status = '1', categoryId = '' } = req.query;
            let query = { _id: { $exists: true } };
            if (status && status !== '') {
                query.status = { $in: Array.isArray(status) ? status : [status] };
            }
            else {
                query.status = '1';
            }
            if (categoryId) {
                query._id = categoryId;
            }
            else {
                query._id = { $exists: true };
            }
            const categories = await category_service_1.default.findAllParentCategories({ query });
            controller.sendSuccessResponse(res, {
                requestedData: categories,
                totalCount: await category_service_1.default.getTotalCount(query),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching categories' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = category_schema_1.categorySchema.safeParse(req.body);
            if (validatedData.success) {
                const { categoryTitle, slug, description, corporateGiftsPriority, categoryType, parentCategory, categoryImageUrl, status } = validatedData.data;
                const user = res.locals.user;
                const category = parentCategory ? await category_service_1.default.findParentCategory(parentCategory) : null;
                var slugData;
                var data = [];
                if (!parentCategory) {
                    data = (0, helpers_1.categorySlugifyManually)(categoryTitle);
                }
                else {
                    data = category?.slug + "-" + (0, helpers_1.categorySlugifyManually)(categoryTitle);
                }
                slugData = data;
                const categoryImage = (req?.file);
                const categoryData = {
                    categoryTitle: (0, helpers_1.capitalizeWords)(categoryTitle),
                    slug: slugData || slug,
                    categoryImageUrl: categoryImageUrl || (0, helpers_1.handleFileUpload)(req, null, (req.file || categoryImage), 'categoryImageUrl', 'category'),
                    description,
                    corporateGiftsPriority,
                    categoryType,
                    parentCategory: parentCategory ? parentCategory : null,
                    level: category?.level ? parseInt(category?.level) + 1 : '0',
                    createdBy: user._id,
                    ['status']: status || '1',
                };
                const newCategory = await category_service_1.default.create(categoryData);
                if (newCategory) {
                    return controller.sendSuccessResponse(res, {
                        requestedData: newCategory,
                        message: 'Category created successfully!'
                    }, 200, {
                        sourceFromId: newCategory._id,
                        sourceFrom: task_log_1.adminTaskLog.serviceManagement.categories,
                        activity: task_log_1.adminTaskLogActivity.create,
                        activityStatus: task_log_1.adminTaskLogStatus.success
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: 'Something went wrong! category cant be inserted. please try again'
                    }, req);
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                }, req);
            }
        }
        catch (error) {
            if (error && error.errors && error.errors.categoryTitle && error.errors.categoryTitle.properties) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        categoryTitle: error.errors.categoryTitle.properties.message
                    }
                }, req);
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while creating Category',
            }, req);
        }
    }
    async findOne(req, res) {
        try {
            const categoryId = req.params.id;
            if (categoryId) {
                const category = await category_service_1.default.findOne(categoryId);
                controller.sendSuccessResponse(res, {
                    requestedData: category,
                    message: 'Success'
                });
            }
            else {
                controller.sendErrorResponse(res, 200, {
                    message: 'Category Id not found!',
                });
            }
        }
        catch (error) {
            controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = category_schema_1.categorySchema.safeParse(req.body);
            if (validatedData.success) {
                const categoryId = req.params.id;
                const user = res.locals.user;
                if (categoryId) {
                    const categoryImage = (req?.file);
                    let { seoData, ...updatedCategoryData } = req.body;
                    const categoryDetails = await category_service_1.default.findOne(categoryId);
                    const parentCategoryDetail = await category_model_1.default.findOne({ _id: updatedCategoryData.parentCategory });
                    updatedCategoryData = {
                        ...updatedCategoryData,
                        categoryTitle: (0, helpers_1.capitalizeWords)(updatedCategoryData.categoryTitle),
                        parentCategory: updatedCategoryData.parentCategory ? updatedCategoryData.parentCategory : null,
                        level: parentCategoryDetail ? Number(parentCategoryDetail.level) + 1 : 0,
                        slug: updatedCategoryData.slug,
                        categoryImageUrl: (0, helpers_1.handleFileUpload)(req, await category_service_1.default.findOne(categoryId), (req.file || categoryImage), 'categoryImageUrl', 'category'),
                        updatedAt: new Date()
                    };
                    if (categoryDetails.parentCategory != updatedCategoryData.parentCategory || updatedCategoryData) {
                        const updatedSlugCategory = await category_service_1.default.update(categoryId, updatedCategoryData);
                        if (updatedSlugCategory) {
                            await category_service_1.default.findSubCategory(updatedSlugCategory);
                        }
                    }
                    if (categoryDetails) {
                        const updatedCategoryMapped = Object.keys(categoryDetails).reduce((mapped, key) => {
                            mapped[key] = categoryDetails[key];
                            return mapped;
                        }, {});
                        return controller.sendSuccessResponse(res, {
                            requestedData: {
                                ...updatedCategoryMapped,
                                message: 'Category updated successfully!'
                            }
                        }, 200, {
                            sourceFromId: categoryDetails._id,
                            sourceFrom: task_log_1.adminTaskLog.serviceManagement.categories,
                            activity: task_log_1.adminTaskLogActivity.update,
                            activityStatus: task_log_1.adminTaskLogStatus.success
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'Category Id not found!',
                        }, req);
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Category Id not found! Please try again with category id',
                    }, req);
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                }, req);
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating category'
            }, req);
        }
    }
    async destroy(req, res) {
        try {
            const categoryId = req.params.id;
            if (categoryId) {
                const category = await category_service_1.default.findOne(categoryId);
                if (category) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Cant to be delete category!!',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This Category details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Category id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting category' });
        }
    }
    async getParentChilledCategory(req, res) {
        try {
            const { page = 1, limit = 10, status = '1' } = req.query;
            const query = { status, _id: { $exists: true } };
            const categories = await category_service_1.default.findAll({ page: parseInt(page), limit: parseInt(limit), query });
            controller.sendSuccessResponse(res, {
                requestedData: categories,
                totalCount: await category_service_1.default.getTotalCount(query),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching categories' });
        }
    }
    async statusChange(req, res) {
        try {
            const validatedData = category_schema_1.categoryStatusSchema.safeParse(req.body);
            if (validatedData.success) {
                const categoryId = req.params.id;
                if (categoryId) {
                    let { status } = req.body;
                    const updatedCategoryData = { status };
                    const updatedCategory = await category_service_1.default.update(categoryId, updatedCategoryData);
                    if (updatedCategory) {
                        return controller.sendSuccessResponse(res, {
                            requestedData: updatedCategory,
                            message: 'Category status updated successfully!'
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'Category Id not found!',
                        }, req);
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Category Id not found! Please try again with Category id',
                    }, req);
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                }, req);
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating Category'
            }, req);
        }
    }
}
exports.default = new CategoryController();
