"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const lodash_1 = require("lodash");
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../base-controller"));
const settings_model_1 = __importDefault(require("../../../model/admin/settings/settings.model"));
const settings_schema_1 = require("../../../utils/schemas/admin/settings/settings.schema");
const controller = new base_controller_1.default();
class SettingsController extends base_controller_1.default {
    async findAll(req, res) {
        try {
            const settings = await settings_model_1.default.find();
            return controller.sendSuccessResponse(res, {
                requestedData: settings,
                totalCount: (0, lodash_1.size)(settings),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching setting' });
        }
    }
    async findOne(req, res) {
        try {
            const settingId = req.params.id;
            if (settingId) {
                const setting = await settings_model_1.default.findById(settingId);
                return controller.sendSuccessResponse(res, {
                    requestedData: setting,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'setting Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async manageSetting(req, res) {
        try {
            const validatedData = await settings_schema_1.settingSchema.safeParseAsync(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 400, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors),
                });
            }
            const data = validatedData.data;
            const filter = { slug: data.slug };
            const update = {
                settingTitle: data.settingTitle,
                settingType: data.settingType,
                settingValues: data.settingValues,
                updatedAt: new Date(),
            };
            const options = { upsert: true, new: true, setDefaultsOnInsert: true };
            const retVal = await settings_model_1.default.findOneAndUpdate(filter, update, options).exec();
            return controller.sendSuccessResponse(res, {
                requestedData: retVal,
                message: `${data.settingTitle} setting managed successfully!`
            }, 200, {
                sourceFromId: retVal?._id,
                sourceFrom: task_log_1.adminTaskLog.settings,
                activity: task_log_1.adminTaskLogActivity.update,
                activityStatus: task_log_1.adminTaskLogStatus.success,
                activityComment: `${data.settingTitle} setting managed successfully!`
            });
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating setting'
            });
        }
    }
}
exports.default = new SettingsController();
