"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const bcrypt_1 = __importDefault(require("bcrypt"));
const user_schema_1 = require("../../../utils/schemas/admin/staff-management/user-schema");
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const admin_user_service_1 = __importDefault(require("../../../services/admin/staff-management/admin-user-service"));
const base_controller_1 = __importDefault(require("../base-controller"));
const designation_model_1 = __importDefault(require("../../../model/admin/staff-management/designation-model"));
const branch_model_1 = __importDefault(require("../../../model/admin/operations/branch-model"));
const department_model_1 = __importDefault(require("../../../model/admin/staff-management/department-model"));
const controller = new base_controller_1.default();
class AdminUserController extends base_controller_1.default {
    async findAll(req, res) {
        try {
            const { designation = '', page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
            let query = { _id: { $exists: true } };
            const userData = await res.locals.user;
            if (status && status !== '') {
                query.status = { $in: Array.isArray(status) ? status : [status] };
            }
            else {
                query.status = '1';
            }
            if (keyword) {
                const keywordRegex = new RegExp(keyword, 'i');
                query = {
                    $or: [
                        { userTitle: keywordRegex },
                        { userEmail: keywordRegex },
                        { userPhoneNumber: keywordRegex }
                    ],
                    ...query
                };
            }
            if (designation) {
                query = {
                    'designation.slug': designation,
                    ...query
                };
            }
            const sort = {};
            if (sortby && sortorder) {
                sort[sortby] = sortorder === 'desc' ? -1 : 1;
            }
            const users = await admin_user_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: users.data,
                totalCount: users.totalCount,
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching users' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = user_schema_1.userSchema.safeParse(req.body);
            if (validatedData.success) {
                const { stateId, cityId, designationCode, userCode, branchCode, departmentCode, userImageUrl, userEmail, userTitle, userPhoneNumber, userPassword } = validatedData.data;
                const user = res.locals.user;
                const designationDetails = await designation_model_1.default.findOne({ designationCode }).select('_id designationCode').lean();
                if (!designationDetails) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Invalid designation code',
                    });
                }
                const branchDetails = await branch_model_1.default.findOne({ branchCode }).select('_id branchCode').lean();
                if (!branchDetails) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'The branch code is not found',
                    });
                }
                const departmentDetails = await department_model_1.default.findOne({ departmentCode }).select('_id departmentCode').lean();
                if (!departmentDetails) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'The department code is not found',
                    });
                }
                const userData = {
                    stateId: stateId !== '' ? stateId : null,
                    cityId: cityId !== '' ? cityId : null,
                    designationId: designationDetails._id,
                    branchId: branchDetails._id,
                    departmentId: departmentDetails._id,
                    userTitle,
                    userCode,
                    userEmail,
                    userImageUrl: req.file ? (0, helpers_1.handleFileUpload)(req, null, req.file, 'userImageUrl', 'adminuser') : userImageUrl,
                    userPhoneNumber,
                    userPassword: await bcrypt_1.default.hash(userPassword, 10),
                    status: '1',
                    createdBy: user._id,
                    createdAt: new Date()
                };
                const newUser = await admin_user_service_1.default.create(userData);
                return controller.sendSuccessResponse(res, {
                    requestedData: newUser,
                    message: 'User created successfully!'
                }, 200, {
                    sourceFromId: newUser?._id,
                    sourceFrom: task_log_1.adminTaskLog.staffManagement.adminUsers,
                    activity: task_log_1.adminTaskLogActivity.create,
                    activityStatus: task_log_1.adminTaskLogStatus.success
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            if (error && error.errors && error.errors.designationId) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        designationId: error.errors.designationId.message
                    }
                });
            }
            if (error && error.errors && error.errors.userEmail) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        userEmail: error.errors.userEmail.message
                    }
                });
            }
            else if (error && error.errors && error.errors.userCode) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        userCode: error.errors.userCode.message
                    }
                });
            }
            else if (error && error.errors && error.errors.userPhoneNumber) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        userPhoneNumber: error.errors.userPhoneNumber.message
                    }
                });
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while creating user'
            });
        }
    }
    async findOne(req, res) {
        try {
            const userId = req.params.id;
            if (userId) {
                const user = await admin_user_service_1.default.findOne(userId);
                return controller.sendSuccessResponse(res, {
                    requestedData: user,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'User Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = user_schema_1.userSchema.safeParse(req.body);
            if (validatedData.success) {
                const userId = req.params.id;
                if (userId) {
                    const user = await admin_user_service_1.default.findOne(userId);
                    const updatedUserData = req.body;
                    const updatedUser = await admin_user_service_1.default.update(userId, {
                        ...updatedUserData,
                        userPassword: user.userPassword === updatedUserData.userPassword ? user.userPassword : await bcrypt_1.default.hash(updatedUserData.userPassword, 10),
                        userImageUrl: (0, helpers_1.handleFileUpload)(req, await admin_user_service_1.default.findOne(userId), req.file, 'userImageUrl', 'adminuser'),
                        updatedAt: new Date()
                    });
                    if (updatedUser) {
                        return controller.sendSuccessResponse(res, {
                            requestedData: updatedUser,
                            message: 'User updated successfully!'
                        }, 200, {
                            sourceFromId: updatedUser._id,
                            sourceFrom: task_log_1.adminTaskLog.staffManagement.adminUsers,
                            activity: task_log_1.adminTaskLogActivity.update,
                            activityStatus: task_log_1.adminTaskLogStatus.success
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'User Id not found!',
                        });
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'User Id not found! Please try again with UserId',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating user'
            });
        }
    }
    async destroy(req, res) {
        try {
            const userId = req.params.id;
            if (userId) {
                const user = await admin_user_service_1.default.findOne(userId);
                if (user) {
                    // await AdminUserService.destroy(userId);
                    // return controller.sendSuccessResponse(res, { message: 'User deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant to delete this user!',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This user details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'User Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting user' });
        }
    }
}
exports.default = new AdminUserController();
