"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../base-controller"));
const department_service_1 = __importDefault(require("../../../services/admin/staff-management/department-service"));
const department_schema_1 = require("../../../utils/schemas/admin/staff-management/department-schema");
const controller = new base_controller_1.default();
class DepartmentController extends base_controller_1.default {
    async findAll(req, res) {
        const { page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { departmentTitle: keywordRegex }
                ],
                ...query
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const department = await department_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: department,
                totalCount: await department_service_1.default.getTotalCount(query),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching department' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = department_schema_1.departmentchema.safeParse(req.body);
            if (validatedData.success) {
                const { departmentTitle, departmentCode, slug } = validatedData.data;
                const user = res.locals.user;
                const departmentData = {
                    departmentTitle,
                    slug: (0, helpers_1.slugifyManually)(departmentTitle),
                    departmentCode,
                    status: '1',
                    createdBy: user._id,
                    createdAt: new Date()
                };
                const newDepartment = await department_service_1.default.create(departmentData);
                return controller.sendSuccessResponse(res, {
                    requestedData: newDepartment,
                    message: 'Department created successfully!'
                }, 200, {
                    sourceFromId: newDepartment._id,
                    sourceFrom: task_log_1.adminTaskLog.staffManagement.department,
                    activity: task_log_1.adminTaskLogActivity.create,
                    activityStatus: task_log_1.adminTaskLogStatus.success
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            if (error && error.errors && error.errors.departmentTitle && error.errors.departmentTitle.properties) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        departmentTitle: error.errors.departmentTitle.properties.message
                    }
                });
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while creating department',
            });
        }
    }
    async findOne(req, res) {
        try {
            const departmentId = req.params.id;
            if (departmentId) {
                const department = await department_service_1.default.findOne(departmentId);
                return controller.sendSuccessResponse(res, {
                    requestedData: department,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Department Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = department_schema_1.departmentchema.safeParse(req.body);
            if (validatedData.success) {
                const departmentId = req.params.id;
                if (departmentId) {
                    const updatedDepartmentData = req.body;
                    const updatedDepartment = await department_service_1.default.update(departmentId, { ...updatedDepartmentData, updatedAt: new Date() });
                    if (updatedDepartment) {
                        return controller.sendSuccessResponse(res, {
                            requestedData: updatedDepartment,
                            message: 'Department updated successfully!'
                        }, 200, {
                            sourceFromId: updatedDepartment._id,
                            sourceFrom: task_log_1.adminTaskLog.staffManagement.department,
                            activity: task_log_1.adminTaskLogActivity.update,
                            activityStatus: task_log_1.adminTaskLogStatus.success
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'Department Id not found!',
                        });
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Department Id not found! Please try again with Department id',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating department'
            });
        }
    }
    async destroy(req, res) {
        try {
            const departmentId = req.params.id;
            if (departmentId) {
                const department = await department_service_1.default.findOne(departmentId);
                if (department) {
                    // await DepartmentService.destroy(departmentId);
                    // return controller.sendSuccessResponse(res, { message: 'Department deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this department',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This department details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Department id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting department' });
        }
    }
}
exports.default = new DepartmentController();
