"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const helpers_1 = require("../../../utils/helpers");
const task_log_1 = require("../../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../base-controller"));
const designation_service_1 = __importDefault(require("../../../services/admin/staff-management/designation-service"));
const designation_schema_1 = require("../../../utils/schemas/admin/staff-management/designation-schema");
const controller = new base_controller_1.default();
class DesignationsController extends base_controller_1.default {
    async findAll(req, res) {
        const { page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { designationTitle: keywordRegex }
                ],
                ...query
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const designations = await designation_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: designations,
                totalCount: await designation_service_1.default.getTotalCount(query),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching designations' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = designation_schema_1.designationSchema.safeParse(req.body);
            if (validatedData.success) {
                const { designationTitle, designationCode, slug } = validatedData.data;
                const user = res.locals.user;
                const designationsData = {
                    designationTitle,
                    designationCode,
                    slug: (0, helpers_1.slugifyManually)(designationTitle),
                    status: '1',
                    createdBy: user._id,
                    createdAt: new Date()
                };
                const newDesignations = await designation_service_1.default.create(designationsData);
                return controller.sendSuccessResponse(res, {
                    requestedData: newDesignations,
                    message: 'Designation created successfully!'
                }, 200, {
                    sourceFromId: newDesignations._id,
                    sourceFrom: task_log_1.adminTaskLog.staffManagement.designations,
                    activity: task_log_1.adminTaskLogActivity.create,
                    activityStatus: task_log_1.adminTaskLogStatus.success
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            if (error && error.errors && error.errors.designationTitle && error.errors.designationTitle.properties) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        designationTitle: error.errors.designationTitle.properties.message
                    }
                });
            }
            else if (error && error.errors && error.errors.designationCode && error.errors.designationCode.properties) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        designationCode: error.errors.designationCode.properties.message
                    }
                });
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while creating designation',
            });
        }
    }
    async findOne(req, res) {
        try {
            const designationsId = req.params.id;
            if (designationsId) {
                const designations = await designation_service_1.default.findOne(designationsId);
                return controller.sendSuccessResponse(res, {
                    requestedData: designations,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Designation Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = designation_schema_1.designationSchema.safeParse(req.body);
            if (validatedData.success) {
                const designationsId = req.params.id;
                if (designationsId) {
                    const updatedDesignationsData = req.body;
                    const updatedDesignations = await designation_service_1.default.update(designationsId, { ...updatedDesignationsData, updatedAt: new Date() });
                    if (updatedDesignations) {
                        return controller.sendSuccessResponse(res, {
                            requestedData: updatedDesignations,
                            message: 'Designation updated successfully!'
                        }, 200, {
                            sourceFromId: updatedDesignations._id,
                            sourceFrom: task_log_1.adminTaskLog.staffManagement.designations,
                            activity: task_log_1.adminTaskLogActivity.update,
                            activityStatus: task_log_1.adminTaskLogStatus.success
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'Designation Id not found!',
                        });
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Designation Id not found! Please try again with Designation id',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating designation'
            });
        }
    }
    async destroy(req, res) {
        try {
            const designationsId = req.params.id;
            if (designationsId) {
                const designations = await designation_service_1.default.findOne(designationsId);
                if (designations) {
                    // await DesignationsService.destroy(designationsId);
                    // return controller.sendSuccessResponse(res, { message: 'Designation deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this designation',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This designation details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Designation id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting designation' });
        }
    }
}
exports.default = new DesignationsController();
