"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const task_log_1 = require("../../constants/admin/task-log");
const helpers_1 = require("../../utils/helpers");
const base_controller_1 = __importDefault(require("../admin/base-controller"));
const country_service_1 = __importDefault(require("../../services/general/country-service"));
const country_schema_1 = require("../../utils/schemas/general/country-schema");
const controller = new base_controller_1.default();
class CountryController extends base_controller_1.default {
    async findAll(req, res) {
        const { page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { countryTitle: keywordRegex }
                ],
                ...query
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const countrys = await country_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: countrys,
                totalCount: await country_service_1.default.getTotalCount(query),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching countrys' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = country_schema_1.countrySchema.safeParse(req.body);
            if (validatedData.success) {
                const { countryTitle, countryCode, currencyCode, countryShortTitle, slug } = validatedData.data;
                const user = res.locals.user;
                const countryData = {
                    countryTitle,
                    slug: (0, helpers_1.slugify)(countryTitle),
                    countryCode,
                    currencyCode,
                    countryShortTitle,
                    status: '1',
                    createdBy: user._id,
                    createdAt: new Date()
                };
                const newCountry = await country_service_1.default.create(countryData);
                return controller.sendSuccessResponse(res, {
                    requestedData: newCountry,
                    message: 'Country created successfully!'
                }, 200, {
                    sourceFromId: newCountry._id,
                    sourceFrom: task_log_1.adminTaskLog.general.country,
                    activity: task_log_1.adminTaskLogActivity.create,
                    activityStatus: task_log_1.adminTaskLogStatus.success
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            if (error && error.errors && error.errors.countryTitle && error.errors.countryTitle.properties) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: {
                        countryTitle: error.errors.countryTitle.properties.message
                    }
                });
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while creating country',
            });
        }
    }
    async findOne(req, res) {
        try {
            const countryId = req.params.id;
            if (countryId) {
                const countrys = await country_service_1.default.findOne(countryId);
                return controller.sendSuccessResponse(res, {
                    requestedData: countrys,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Country Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = country_schema_1.countrySchema.safeParse(req.body);
            if (validatedData.success) {
                const countryId = req.params.id;
                if (countryId) {
                    const updatedCountryData = req.body;
                    const updatedCountrys = await country_service_1.default.update(countryId, { ...updatedCountryData, updatedAt: new Date() });
                    if (updatedCountrys) {
                        return controller.sendSuccessResponse(res, {
                            requestedData: updatedCountrys,
                            message: 'Country updated successfully!'
                        }, 200, {
                            sourceFromId: updatedCountrys._id,
                            sourceFrom: task_log_1.adminTaskLog.general.country,
                            activity: task_log_1.adminTaskLogActivity.update,
                            activityStatus: task_log_1.adminTaskLogStatus.success
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'Country Id not found!',
                        });
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Country Id not found! Please try again with Country id',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating country'
            });
        }
    }
    async destroy(req, res) {
        try {
            const countryId = req.params.id;
            if (countryId) {
                const countrys = await country_service_1.default.findOne(countryId);
                if (countrys) {
                    // await CountryService.destroy(countryId);
                    // return controller.sendSuccessResponse(res, { message: 'Country deleted successfully!' });
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this country',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This country details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Country id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting country' });
        }
    }
}
exports.default = new CountryController();
