"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const helpers_1 = require("../../utils/helpers");
const task_log_1 = require("../../constants/admin/task-log");
const base_controller_1 = __importDefault(require("../admin/base-controller"));
const vendor_schema_1 = require("../../utils/schemas/outsourced/vendor-schema");
const vendors_service_1 = __importDefault(require("../../services/outsourced/vendors-service"));
const controller = new base_controller_1.default();
class VendorController extends base_controller_1.default {
    async findAll(req, res) {
        const { page_size = 1, limit = 10, status = ['0', '1', '2'], sortby = '', sortorder = '', keyword = '' } = req.query;
        let query = { _id: { $exists: true } };
        if (status && status !== '') {
            query.status = { $in: Array.isArray(status) ? status : [status] };
        }
        else {
            query.status = '1';
        }
        if (keyword) {
            const keywordRegex = new RegExp(keyword, 'i');
            query = {
                $or: [
                    { vendorTitle: keywordRegex }
                ],
                ...query
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const vendor = await vendors_service_1.default.findAll({
                page: parseInt(page_size),
                limit: parseInt(limit),
                query,
                sort
            });
            return controller.sendSuccessResponse(res, {
                requestedData: vendor,
                totalCount: await vendors_service_1.default.getTotalCount(query),
                message: 'Success!'
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching vendor' });
        }
    }
    async create(req, res) {
        try {
            const validatedData = vendor_schema_1.vendorSchema.safeParse(req.body);
            if (validatedData.success) {
                const { vendorTitle, vendorCode, contactName, contactEmail, contactPhone, address, city, country, postalCode, vendorType, notes } = validatedData.data;
                const user = res.locals.user;
                const vendorData = {
                    vendorTitle,
                    slug: (0, helpers_1.slugifyManually)(vendorTitle),
                    vendorCode,
                    contactName,
                    contactEmail,
                    contactPhone,
                    address,
                    city,
                    country,
                    postalCode,
                    vendorType,
                    notes,
                    status: '1',
                    createdBy: user._id,
                    createdAt: new Date()
                };
                const newVendor = await vendors_service_1.default.create(vendorData);
                return controller.sendSuccessResponse(res, {
                    requestedData: newVendor,
                    message: 'Vendor created successfully!'
                }, 200, {
                    sourceFromId: newVendor._id,
                    sourceFrom: task_log_1.adminTaskLog.outsourced.vendors,
                    activity: task_log_1.adminTaskLogActivity.create,
                    activityStatus: task_log_1.adminTaskLogStatus.success
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            const { errors } = error || {};
            const validationErrors = {};
            if (errors) {
                Object.keys(errors).forEach((key) => {
                    validationErrors[key] = errors[key].message;
                });
                if (Object.keys(validationErrors).length > 0) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Validation error',
                        validation: validationErrors
                    });
                }
            }
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while creating vendor',
            });
        }
    }
    async findOne(req, res) {
        try {
            const vendorId = req.params.id;
            if (vendorId) {
                const vendor = await vendors_service_1.default.findOne(vendorId);
                return controller.sendSuccessResponse(res, {
                    requestedData: vendor,
                    message: 'Success'
                });
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Vendor Id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message });
        }
    }
    async update(req, res) {
        try {
            const validatedData = vendor_schema_1.vendorSchema.safeParse(req.body);
            if (validatedData.success) {
                const vendorId = req.params.id;
                if (vendorId) {
                    const updatedVendorData = req.body;
                    const updatedVendor = await vendors_service_1.default.update(vendorId, { ...updatedVendorData, updatedAt: new Date() });
                    if (updatedVendor) {
                        return controller.sendSuccessResponse(res, {
                            requestedData: updatedVendor,
                            message: 'Vendor updated successfully!'
                        }, 200, {
                            sourceFromId: updatedVendor._id,
                            sourceFrom: task_log_1.adminTaskLog.outsourced.vendors,
                            activity: task_log_1.adminTaskLogActivity.update,
                            activityStatus: task_log_1.adminTaskLogStatus.success
                        });
                    }
                    else {
                        return controller.sendErrorResponse(res, 200, {
                            message: 'Vendor Id not found!',
                        });
                    }
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'Vendor Id not found! Please try again with Vendor id',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, {
                message: error.message || 'Some error occurred while updating vendor'
            });
        }
    }
    async destroy(req, res) {
        try {
            const vendorId = req.params.id;
            if (vendorId) {
                const vendor = await vendors_service_1.default.findOne(vendorId);
                if (vendor) {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'You cant delete this vendor',
                    });
                }
                else {
                    return controller.sendErrorResponse(res, 200, {
                        message: 'This vendor details not found!',
                    });
                }
            }
            else {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Vendor id not found!',
                });
            }
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while deleting vendor' });
        }
    }
}
exports.default = new VendorController();
