"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const base_controller_1 = __importDefault(require("../admin/base-controller"));
const technician_service_1 = __importDefault(require("../../services/technician/technician-service"));
const project_utils_1 = require("../../utils/project-utils");
const job_order_assignment_history_model_1 = __importDefault(require("../../model/admin/job-order/job-order-assignment-history-model"));
const lodash_1 = require("lodash");
const job_order_schema_1 = require("../../utils/schemas/admin/job-order/job-order-schema");
const helpers_1 = require("../../utils/helpers");
const job_order_1 = require("../../constants/admin/job-order");
const job_order_details_model_1 = __importDefault(require("../../model/admin/job-order/job-order-details-model"));
const controller = new base_controller_1.default();
class TechnicianController extends base_controller_1.default {
    async findAll(req, res) {
        const { technicianId = '', page_size = 1, limit = 10, status = '', jobOrderDetailId = '', technicianStatus = '', sortby = '', sortorder = '', keyword = '' } = req.query;
        const user = res.locals.user;
        let query = { _id: { $exists: true } };
        if (technicianStatus) {
        }
        if (jobOrderDetailId) {
            query.jobOrderDetailId = new mongoose_1.default.Types.ObjectId(jobOrderDetailId);
        }
        const visitorBranchId = (0, project_utils_1.getVisitorBranchId)(user);
        const contextTechnicianId = (0, project_utils_1.getTechnicianId)(user);
        if (technicianId) {
            query.technicianId = new mongoose_1.default.Types.ObjectId(technicianId);
        }
        else if (contextTechnicianId) {
            query.$or = [
                { 'technicianId': new mongoose_1.default.Types.ObjectId(contextTechnicianId) },
                { 'reassignedFromId': new mongoose_1.default.Types.ObjectId(contextTechnicianId) },
            ];
        }
        else if (visitorBranchId) {
            query.$or = [
                { 'jobOrderDetails.branchId': new mongoose_1.default.Types.ObjectId(visitorBranchId) },
                { 'jobOrderDetails.transferBranchId': new mongoose_1.default.Types.ObjectId(visitorBranchId) },
            ];
        }
        if (keyword) {
            const escapedKeyword = keyword.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
            const keywordRegex = new RegExp(`^${escapedKeyword}`, 'i');
            query = {
                $or: [
                    { taskTitle: { $regex: keywordRegex } },
                    { description: { $regex: keywordRegex } },
                ],
                ...query
            };
        }
        const sort = {};
        if (sortby && sortorder) {
            sort[sortby] = sortorder === 'desc' ? -1 : 1;
        }
        try {
            const assignments = await technician_service_1.default.findAll({
                query,
                page: parseInt(page_size),
                limit: parseInt(limit),
                sort,
            });
            return controller.sendSuccessResponse(res, {
                requestedData: assignments[0]?.data,
                totalCount: assignments[0]?.totalCount[0]?.total,
                message: 'Success!',
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching job order assignments' });
        }
    }
    async technicianAcceptChange(req, res) {
        try {
            const taskId = req.params.id;
            if (!taskId) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Task id not found!',
                });
            }
            const validatedData = job_order_schema_1.technicianAcceptStatusChangeJobOrderSchema.safeParse(req.body);
            if (!validatedData.success) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Validation error',
                    validation: (0, helpers_1.formatZodError)(validatedData.error.errors)
                });
            }
            const taskDetails = await job_order_assignment_history_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(taskId) });
            if ((0, lodash_1.isEmpty)(taskDetails)) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Task details not found!',
                });
            }
            const { orderDetailId, technicianStatus } = validatedData.data;
            if (taskDetails.technicianStatus === technicianStatus && technicianStatus === job_order_1.technicianStatusListJson.acknowledged) {
                return controller.sendErrorResponse(res, 200, {
                    message: `The technician status is already set to '${(0, lodash_1.capitalize)(technicianStatus)}'. No further updates are required.`,
                });
            }
            if (taskDetails.technicianStatus === job_order_1.technicianStatusListJson.acknowledged && technicianStatus === job_order_1.technicianStatusListJson.pending) {
                return controller.sendErrorResponse(res, 200, {
                    message: `The technician status is already set to '${(0, lodash_1.capitalize)(job_order_1.technicianStatusListJson.acknowledged)}'. No further updates are required.`,
                });
            }
            const retVal = await job_order_assignment_history_model_1.default.findByIdAndUpdate(taskDetails._id, {
                technicianStatus,
                technicianStatusAt: new Date()
            }, { new: true, useFindAndModify: false });
            if ((0, lodash_1.isEmpty)(retVal)) {
                return controller.sendErrorResponse(res, 200, {
                    message: 'Something went wrong on task updation!',
                });
            }
            await job_order_details_model_1.default.findByIdAndUpdate(taskDetails.jobOrderDetailId, {
                technicianStatus,
                technicianStatusAt: new Date()
            }, { new: true, useFindAndModify: false });
            return controller.sendSuccessResponse(res, {
                requestedData: retVal,
                message: 'Task accepted successfully completed!',
            }, 200);
        }
        catch (error) {
            return controller.sendErrorResponse(res, 500, { message: error.message || 'Some error occurred while fetching job order assignments' });
        }
    }
}
exports.default = new TechnicianController();
