"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const pagination_1 = require("../../../components/pagination");
const branch_config_1 = require("../../../utils/config/admin/operation/branch-config");
const general_config_1 = require("../../../utils/config/general/general-config");
const customer_config_1 = require("../../../utils/config/admin/customerer-managemet/customer-config");
const customer_master_model_1 = __importDefault(require("../../../model/admin/customer-management/customer-master-model"));
class Customerinventory {
    async findAll(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        let pipeline = [
            { $match: query },
            { $sort: finalSort },
            { $skip: skip },
            { $limit: limit },
            (0, branch_config_1.branchLookup)(),
            general_config_1.countriesLookup,
            general_config_1.statesLookup,
            general_config_1.citiesLookup,
            ...customer_config_1.customerProject,
        ];
        return customer_master_model_1.default.aggregate(pipeline).exec();
    }
    async getTotalCount(query = {}) {
        try {
            const totalCount = await customer_master_model_1.default.countDocuments(query);
            return totalCount;
        }
        catch (error) {
            throw new Error('Error fetching total count of customers: ' + error?.message);
        }
    }
    async create(customerData) {
        const createdCustomer = await customer_master_model_1.default.create(customerData);
        if (createdCustomer) {
            const pipeline = [
                { $match: { _id: createdCustomer._id } },
                (0, branch_config_1.branchLookup)(),
                general_config_1.countriesLookup,
                general_config_1.statesLookup,
                general_config_1.citiesLookup,
                ...customer_config_1.customerProject,
            ];
            const [customerWithBranch] = await customer_master_model_1.default.aggregate(pipeline).exec();
            return customerWithBranch;
        }
        else {
            return null;
        }
    }
    async findOne(customerId) {
        const pipeline = [
            { $match: { _id: new mongoose_1.default.Types.ObjectId(customerId) } },
            (0, branch_config_1.branchLookup)(),
            general_config_1.countriesLookup,
            general_config_1.statesLookup,
            general_config_1.citiesLookup,
            ...customer_config_1.customerProject,
        ];
        const [customerWithDetails] = await customer_master_model_1.default.aggregate(pipeline).exec();
        return customerWithDetails || null;
    }
    async update(customerId, customerData) {
        const updatdCustomer = await customer_master_model_1.default.findByIdAndUpdate(customerId, customerData, { new: true, useFindAndModify: false });
        if (updatdCustomer) {
            const pipeline = [
                { $match: { _id: customerId } },
                (0, branch_config_1.branchLookup)(),
                general_config_1.countriesLookup,
                general_config_1.statesLookup,
                general_config_1.citiesLookup,
                ...customer_config_1.customerProject,
            ];
            const [customerWithBranch] = await customer_master_model_1.default.aggregate(pipeline).exec();
            return customerWithBranch;
        }
        else {
            return null;
        }
    }
    async destroy(customerId) {
        return customer_master_model_1.default.findOneAndDelete({ _id: customerId });
    }
}
exports.default = new Customerinventory();
