"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const pagination_1 = require("../../../components/pagination");
const helpers_1 = require("../../../utils/helpers");
const attribute_master_model_1 = __importDefault(require("../../../model/admin/inventory-management/attribute-master-model"));
const attribute_config_1 = require("../../../utils/config/admin/inventory-management/attribute-config");
const attribute_details_model_1 = __importDefault(require("../../../model/admin/inventory-management/attribute-details-model"));
class AttributesService {
    async findAll(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        let pipeline = [
            { $match: query },
            { $sort: finalSort },
            { $skip: skip },
            { $limit: limit },
            attribute_config_1.attributeDetailsLookup,
            attribute_config_1.attributeProject
        ];
        return attribute_master_model_1.default.aggregate(pipeline).exec();
    }
    async getTotalCount(query = {}) {
        try {
            const totalCount = await attribute_master_model_1.default.countDocuments(query);
            return totalCount;
        }
        catch (error) {
            throw new Error('Error fetching total count of attribute');
        }
    }
    async create(attributeData) {
        const createdAttribute = await attribute_master_model_1.default.create(attributeData);
        if (createdAttribute) {
            const pipeline = [
                { $match: { _id: createdAttribute._id } },
                attribute_config_1.attributeDetailsLookup,
                attribute_config_1.attributeProject
            ];
            const createdAttributeWithValues = await attribute_master_model_1.default.aggregate(pipeline);
            return createdAttributeWithValues[0];
        }
        else {
            return null;
        }
    }
    async findOne(attributeId) {
        if (attributeId) {
            const objectId = new mongoose_1.default.Types.ObjectId(attributeId);
            const pipeline = [
                { $match: { _id: objectId } },
                attribute_config_1.attributeDetailsLookup,
                attribute_config_1.attributeProject
            ];
            const attributeDataWithValues = await attribute_master_model_1.default.aggregate(pipeline);
            return attributeDataWithValues[0];
        }
        else {
            return null;
        }
    }
    async findOneAttributeFromExcel(attributeKeyValue) {
        const { attributeTitle, attributeType } = attributeKeyValue;
        if (attributeTitle && attributeType) {
            const resultAttribute = await attribute_master_model_1.default.findOne({ attributeTitle: (0, helpers_1.capitalizeWords)(attributeTitle), attributeType });
            if (resultAttribute) {
                const attributeDetailResult = await this.findOneAttributeDetail(attributeKeyValue, resultAttribute._id);
                const result = {
                    attributeId: resultAttribute._id,
                    attributeDetailId: attributeDetailResult._id
                };
                return result;
            }
            else {
                const attributeData = {
                    attributeTitle: (0, helpers_1.capitalizeWords)(attributeTitle),
                    attributeType: attributeType,
                    isExcel: true,
                    slug: (0, helpers_1.slugify)(attributeTitle)
                };
                const attributeResult = await this.create(attributeData);
                if (attributeResult) {
                    const attributeDetailResult = await this.findOneAttributeDetail(attributeKeyValue, attributeResult._id);
                    const result = {
                        attributeId: attributeResult._id,
                        attributeDetailId: attributeDetailResult._id
                    };
                    return result;
                }
            }
        }
    }
    async findOneAttributeDetail(attributeData, attributeId) {
        const resultAttribute = await attribute_details_model_1.default.findOne({ $and: [{ itemName: attributeData.attributeItemName }, { attributeId: attributeId }] });
        if (resultAttribute) {
            return resultAttribute;
        }
        else {
            const insertAttributeDetail = {
                attributeId: attributeId,
                itemName: attributeData.attributeItemName,
                itemValue: attributeData.attributeItemValue,
            };
            const attributeDetailsResult = await attribute_details_model_1.default.create(insertAttributeDetail);
            if (attributeDetailsResult) {
                return attributeDetailsResult;
            }
        }
    }
    async update(attributeId, attributeData) {
        const updatedAttributes = await attribute_master_model_1.default.findByIdAndUpdate(attributeId, attributeData, { new: true, useFindAndModify: false });
        if (updatedAttributes) {
            const pipeline = [
                { $match: { _id: updatedAttributes._id } },
                attribute_config_1.attributeDetailsLookup,
                attribute_config_1.attributeProject
            ];
            const updatedAttributesWithValues = await attribute_master_model_1.default.aggregate(pipeline);
            return updatedAttributesWithValues[0];
        }
        else {
            return null;
        }
    }
    async attributeDetailsService(attributeId, attributeDetails) {
        try {
            if (attributeId) {
                const existingEntries = await attribute_details_model_1.default.find({ attributeId: attributeId });
                if (existingEntries) {
                    const attributeDetailIDsToRemove = existingEntries
                        .filter(entry => !attributeDetails.some((data) => data?._id?.toString() === entry?._id?.toString()))
                        .map(entry => entry._id);
                    await attribute_details_model_1.default.deleteMany({ attributeId: attributeId, _id: { $in: attributeDetailIDsToRemove } });
                }
                const inventryPricingPromises = await Promise.all(attributeDetails.map(async (data) => {
                    const existingEntry = await attribute_details_model_1.default.findOne({ _id: data._id });
                    if (existingEntry) {
                        await attribute_details_model_1.default.findByIdAndUpdate(existingEntry._id, { ...data, attributeId: attributeId });
                    }
                    else {
                        await attribute_details_model_1.default.create({ ...data, attributeId: attributeId });
                    }
                }));
                await Promise.all(inventryPricingPromises);
                return await attribute_details_model_1.default.find({ attributeId: attributeId });
            }
            else {
                throw 'Could not find attribute Id';
            }
        }
        catch (error) {
            console.error('Error in attributeDetailsService:', error);
            throw error;
        }
    }
    async destroy(attributeId) {
        return attribute_master_model_1.default.findOneAndDelete({ _id: attributeId });
    }
}
exports.default = new AttributesService();
