"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const pagination_1 = require("../../../../components/pagination");
const collections_1 = require("../../../../constants/collections");
const item_details_model_1 = __importDefault(require("../../../../model/admin/inventory-management/items/item-details-model"));
const items_config_1 = require("../../../../utils/config/admin/inventory-management/items-config");
const warranty_config_1 = require("../../../../utils/config/admin/inventory-management/warranty-config");
const mongoose_1 = __importDefault(require("mongoose"));
class ItemDetailsInventoryService {
    async findAllItem(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const { itemMasterSlug = '', branchStockQuery, getstockbranch = '0' } = options;
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        // console.log('query', JSON.stringify(query, null, 2));
        let pipeline = [
            // itemMasterLookup,
            {
                $lookup: {
                    from: `${collections_1.collections.inventoryManagement.items.itemsMaster}`,
                    localField: 'itemMasterId',
                    foreignField: '_id',
                    as: 'itemMaster',
                    ...(itemMasterSlug ? {
                        pipeline: [
                            {
                                $match: {
                                    'slug': itemMasterSlug.toString()
                                }
                            }
                        ]
                    } : {})
                }
            },
            {
                $unwind: {
                    path: "$itemMaster",
                    preserveNullAndEmptyArrays: false
                }
            },
            ...(0, items_config_1.itemStockInventoryLookup)(branchStockQuery, getstockbranch),
            { $match: query },
            { $sort: finalSort },
        ];
        // if (itemMasterSlug) {
        //     pipeline.push({
        //         $match: {
        //             'itemMaster.slug': itemMasterSlug.toString()
        //         }
        //     });
        // }
        const itemsPipeline = [
            ...pipeline,
            ...(skip ? [{ $skip: skip }] : []),
            ...(limit ? [{ $limit: limit }] : []),
            warranty_config_1.warrantyLookup,
            ...items_config_1.itemDetailsProject,
        ];
        const totalCountPipeline = [
            ...pipeline,
            { $count: "count" }
        ];
        const facetPipeline = [
            {
                $facet: {
                    items: itemsPipeline,
                    totalCount: totalCountPipeline
                }
            },
            {
                $unwind: {
                    path: "$totalCount",
                    preserveNullAndEmptyArrays: true
                }
            },
            {
                $addFields: {
                    totalCount: { $ifNull: ["$totalCount.count", 0] }
                }
            }
        ];
        const [result] = await item_details_model_1.default.aggregate(facetPipeline).exec();
        return {
            items: result?.items || [],
            totalCount: result?.totalCount || 0
        };
    }
    async getTotalCountItem(query = {}) {
        try {
            const totalCount = await item_details_model_1.default.countDocuments(query);
            return totalCount;
        }
        catch (error) {
            throw new Error('Error fetching total count of item detail');
        }
    }
    async createItem(itemDetailData) {
        try {
            const createdItemDetail = await item_details_model_1.default.create(itemDetailData);
            if (createdItemDetail) {
                const pipeline = [
                    { $match: { _id: createdItemDetail._id } },
                    items_config_1.itemMasterLookup,
                    warranty_config_1.warrantyLookup,
                    ...items_config_1.itemDetailsProject
                ];
                const [itemDetailWithBranch] = await item_details_model_1.default.aggregate(pipeline).exec();
                return itemDetailWithBranch;
            }
            else {
                return null;
            }
        }
        catch (error) {
            console.error('Error in create method:', error);
            throw error;
        }
    }
    async findOneItem(itemId) {
        const pipeline = [
            { $match: { _id: new mongoose_1.default.Types.ObjectId(itemId) } },
            items_config_1.itemMasterLookup,
            warranty_config_1.warrantyLookup,
            ...items_config_1.itemDetailsProject
        ];
        const [itemDetailWithBranch] = await item_details_model_1.default.aggregate(pipeline).exec();
        return itemDetailWithBranch;
    }
    async updateItem(itemId, itemDetailData) {
        const updatdItemDetail = await item_details_model_1.default.findByIdAndUpdate(itemId, itemDetailData, { new: true, useFindAndModify: false });
        if (updatdItemDetail) {
            const pipeline = [
                { $match: { _id: new mongoose_1.default.Types.ObjectId(itemId) } },
                items_config_1.itemMasterLookup,
                warranty_config_1.warrantyLookup,
                ...items_config_1.itemDetailsProject
            ];
            const [itemDetailWithBranch] = await item_details_model_1.default.aggregate(pipeline).exec();
            return itemDetailWithBranch;
        }
        else {
            return null;
        }
    }
    async destroyItem(itemId) {
        return item_details_model_1.default.findOneAndDelete({ _id: new mongoose_1.default.Types.ObjectId(itemId) });
    }
}
exports.default = new ItemDetailsInventoryService();
