"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const pagination_1 = require("../../../../components/pagination");
const item_master_attribute_model_1 = __importDefault(require("../../../../model/admin/inventory-management/items/item-master-attribute-model"));
const items_master_model_1 = __importDefault(require("../../../../model/admin/inventory-management/items/items-master-model"));
const category_config_1 = require("../../../../utils/config/admin/inventory-management/category-config");
const items_config_1 = require("../../../../utils/config/admin/inventory-management/items-config");
const lodash_1 = require("lodash");
class ItemsMasterInventoryService {
    async findAll(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        let pipeline = [
            { $match: query },
            { $sort: finalSort },
        ];
        const facetPipeline = [
            {
                $facet: {
                    totalCount: [
                        { $count: 'total' }
                    ],
                    data: [
                        { $skip: skip },
                        { $limit: limit },
                        ...((query['categoryId'] === '' || query['categoryId'] === undefined) ? [
                            category_config_1.categoryLookup,
                            { $unwind: { path: "$category", preserveNullAndEmptyArrays: true } }
                        ] : []),
                        ...items_config_1.itemMasterAttributes,
                        ...items_config_1.itemMasterProject,
                    ]
                }
            },
            {
                $project: {
                    totalCount: { $ifNull: [{ $arrayElemAt: ['$totalCount.total', 0] }, 0] },
                    data: 1
                }
            }
        ];
        if (query['categoryId']) {
            pipeline.push(category_config_1.categoryLookup, { $unwind: { path: "$category", preserveNullAndEmptyArrays: true } });
        }
        pipeline.push(...facetPipeline);
        const retVal = await items_master_model_1.default.aggregate(pipeline).exec();
        return {
            data: (0, lodash_1.get)(retVal, '[0].data', []),
            totalCount: (0, lodash_1.get)(retVal, '[0].totalCount', 0)
        };
    }
    async getTotalCount(query = {}) {
        try {
            const totalCount = await items_master_model_1.default.countDocuments(query);
            return totalCount;
        }
        catch (error) {
            throw new Error('Error fetching total count of itemMaster');
        }
    }
    async create(itemMasterData) {
        try {
            const createdItemMaster = await items_master_model_1.default.create(itemMasterData);
            if (createdItemMaster) {
                const pipeline = [
                    { $match: { _id: createdItemMaster._id } },
                    category_config_1.categoryLookup,
                    { $unwind: { path: "$category", preserveNullAndEmptyArrays: true } },
                    ...items_config_1.itemMasterAttributes,
                    ...items_config_1.itemMasterProject,
                ];
                const [itemMasterWithBranch] = await items_master_model_1.default.aggregate(pipeline).exec();
                return itemMasterWithBranch;
            }
            else {
                return null;
            }
        }
        catch (error) {
            console.error('Error in create method:', error);
            throw error;
        }
    }
    async findOne(itemMasterId, branchId) {
        const pipeline = [
            { $match: { _id: new mongoose_1.default.Types.ObjectId(itemMasterId) } },
            category_config_1.categoryLookup,
            { $unwind: { path: "$category", preserveNullAndEmptyArrays: true } },
            (0, items_config_1.itemMasterWithDetail)(branchId),
            ...items_config_1.itemMasterAttributes,
        ];
        const [itemMasterWithBranch] = await items_master_model_1.default.aggregate(pipeline).exec();
        return itemMasterWithBranch;
    }
    async update(itemMasterId, itemMasterData) {
        const updatdItemMaster = await items_master_model_1.default.findByIdAndUpdate(itemMasterId, itemMasterData, { new: true, useFindAndModify: false });
        if (updatdItemMaster) {
            const pipeline = [
                { $match: { _id: new mongoose_1.default.Types.ObjectId(itemMasterId) } },
                category_config_1.categoryLookup,
                { $unwind: { path: "$category", preserveNullAndEmptyArrays: true } },
                ...items_config_1.itemMasterProject,
            ];
            const [itemMasterWithBranch] = await items_master_model_1.default.aggregate(pipeline).exec();
            return itemMasterWithBranch;
        }
        else {
            return null;
        }
    }
    async destroy(itemMasterId) {
        return items_master_model_1.default.findOneAndDelete({ _id: new mongoose_1.default.Types.ObjectId(itemMasterId) });
    }
    async createItemMasterAttributes(itemMasterAttributes) {
        try {
            if (!itemMasterAttributes) {
                return null;
            }
            const { itemMasterId, attributeId, attributeDetailId } = itemMasterAttributes;
            const filter = {
                itemMasterId,
                attributeId,
                attributeDetailId,
            };
            const update = {
                $set: {
                    itemMasterId,
                    attributeId,
                    attributeDetailId,
                    createdAt: new Date(),
                },
            };
            const options = {
                upsert: true,
                new: true,
            };
            const attributeValue = await item_master_attribute_model_1.default.findOneAndUpdate(filter, update, options);
            return attributeValue;
        }
        catch (error) {
            // Handle any potential error (optional logging)
            return null;
        }
    }
}
exports.default = new ItemsMasterInventoryService();
