"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const lodash_1 = require("lodash");
const pagination_1 = require("../../../../components/pagination");
const category_config_1 = require("../../../../utils/config/admin/inventory-management/category-config");
const items_config_1 = require("../../../../utils/config/admin/inventory-management/items-config");
const products_attribute_model_1 = __importDefault(require("../../../../model/admin/inventory-management/products/products-attribute.model"));
const products_model_1 = __importDefault(require("../../../../model/admin/inventory-management/products/products.model"));
const products_stock_inventory_model_1 = __importDefault(require("../../../../model/admin/inventory-management/products/products-stock-inventory-model"));
const collections_1 = require("../../../../constants/collections");
class ProductsInventoryService {
    constructor(productsModel = products_model_1.default, productAttributesModel = products_attribute_model_1.default, productStockInventoryModel = products_stock_inventory_model_1.default) {
        this.productsModel = productsModel;
        this.productAttributesModel = productAttributesModel;
        this.productStockInventoryModel = productStockInventoryModel;
    }
    async findAll(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const { isNotFetchCategory = '' } = options;
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        let pipeline = [
            { $match: query },
            { $sort: finalSort },
        ];
        const facetPipeline = [
            {
                $facet: {
                    totalCount: [
                        { $count: 'total' }
                    ],
                    data: [
                        { $skip: skip },
                        { $limit: limit },
                        ...(isNotFetchCategory !== '1' ? [
                            category_config_1.categoryLookup,
                            { $unwind: { path: "$category", preserveNullAndEmptyArrays: true } }
                        ] : []),
                    ]
                }
            },
            {
                $project: {
                    totalCount: { $ifNull: [{ $arrayElemAt: ['$totalCount.total', 0] }, 0] },
                    data: 1
                }
            }
        ];
        pipeline.push(...facetPipeline);
        const retVal = await this.productsModel.aggregate(pipeline).exec();
        return {
            data: (0, lodash_1.get)(retVal, '[0].data', []),
            totalCount: (0, lodash_1.get)(retVal, '[0].totalCount', 0)
        };
    }
    async findAllFilterData(options = {}) {
        const { query, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        let pipeline = [
            { $match: query },
            { $sort: finalSort },
        ];
        const facetPipeline = [
            {
                $facet: {
                    totalCount: [
                        { $count: 'total' }
                    ],
                    data: []
                }
            },
            {
                $project: {
                    totalCount: { $ifNull: [{ $arrayElemAt: ['$totalCount.total', 0] }, 0] },
                    data: 1
                }
            }
        ];
        pipeline.push(...facetPipeline);
        const retVal = await this.productsModel.aggregate(pipeline).exec();
        return {
            data: (0, lodash_1.get)(retVal, '[0].data', []),
            totalCount: (0, lodash_1.get)(retVal, '[0].totalCount', 0)
        };
    }
    async getStockInventory(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        let pipeline = [
            { $match: query },
            { $sort: finalSort },
        ];
        const facetPipeline = [
            {
                $facet: {
                    totalCount: [
                        { $count: 'total' }
                    ],
                    data: [
                        { $skip: skip },
                        { $limit: limit },
                        {
                            $lookup: {
                                from: `${collections_1.collections.inventoryManagement.products.productStockInventory}`,
                                localField: '_id',
                                foreignField: 'productId',
                                as: 'stockInventory',
                                pipeline: [
                                    {
                                        $lookup: {
                                            from: `${collections_1.collections.operations.branches}`,
                                            localField: 'branchId',
                                            foreignField: '_id',
                                            as: 'branch'
                                        }
                                    },
                                    { $unwind: { path: "$branch", preserveNullAndEmptyArrays: true } },
                                ]
                            }
                        },
                        {
                            $lookup: {
                                from: `${collections_1.collections.inventoryManagement.products.productsSerialNumbers}`,
                                localField: '_id',
                                foreignField: 'productId',
                                as: 'productsSerialNumbers',
                            }
                        },
                        {
                            $set: {
                                productsSerialNumbers: {
                                    $slice: [
                                        {
                                            $sortArray: {
                                                input: "$productsSerialNumbers",
                                                sortBy: { createdAt: -1 }
                                            }
                                        },
                                        5
                                    ]
                                }
                            }
                        },
                        {
                            $project: {
                                _id: 1,
                                itemCode: 1,
                                itemTitle: 1,
                                slug: 1,
                                barCode: 1,
                                itemQuantity: 1,
                                itemPrice: 1,
                                itemImageUrl: 1,
                                itemDescription: 1,
                                stockInventory: 1,
                                productsSerialNumbers: 1,
                            }
                        }
                    ]
                }
            },
            { $project: { totalCount: { $ifNull: [{ $arrayElemAt: ['$totalCount.total', 0] }, 0] }, data: 1 } }
        ];
        pipeline.push(...facetPipeline);
        const retVal = await this.productsModel.aggregate(pipeline).exec();
        return {
            data: (0, lodash_1.get)(retVal, '[0].data', []),
            totalCount: (0, lodash_1.get)(retVal, '[0].totalCount', 0)
        };
    }
    async getTotalCount(query = {}) {
        try {
            const totalCount = await this.productsModel.countDocuments(query);
            return totalCount;
        }
        catch (error) {
            throw new Error('Error fetching total count of itemMaster');
        }
    }
    async create(itemMasterData) {
        try {
            const createdItemMaster = await this.productsModel.findByIdAndUpdate(itemMasterData, { new: true, useFindAndModify: false });
            return createdItemMaster;
        }
        catch (error) {
            console.error('Error in create method:', error);
            throw error;
        }
    }
    async manageProduct(itemMasterData) {
        try {
            const { itemCode, ...restValues } = itemMasterData;
            const filter = { itemCode };
            const update = {
                $set: { ...restValues, itemCode },
                $setOnInsert: { createdAt: new Date(), },
            };
            const options = { upsert: true, new: true, setDefaultsOnInsert: true, };
            const createdItemMaster = await this.productsModel.findOneAndUpdate(filter, update, options).exec();
            return createdItemMaster;
        }
        catch (error) {
            console.error('Error in create method:', error);
            throw error;
        }
    }
    async findOne(itemMasterId, branchId) {
        const pipeline = [
            { $match: { _id: new mongoose_1.default.Types.ObjectId(itemMasterId) } },
            category_config_1.categoryLookup,
            { $unwind: { path: "$category", preserveNullAndEmptyArrays: true } },
            ...items_config_1.itemMasterAttributes,
        ];
        const [itemMasterWithBranch] = await this.productsModel.aggregate(pipeline).exec();
        return itemMasterWithBranch;
    }
    async update(itemMasterId, itemMasterData) {
        try {
            const updatdItemMaster = await this.productsModel.findByIdAndUpdate(itemMasterId, itemMasterData, { new: true, useFindAndModify: false });
            return updatdItemMaster;
        }
        catch (error) {
            console.error('Error in create method:', error);
            throw error;
        }
    }
    async destroy(itemMasterId) {
        return this.productsModel.findOneAndDelete({ _id: new mongoose_1.default.Types.ObjectId(itemMasterId) });
    }
    async createProductAttributes(productAttributes) {
        try {
            if (!productAttributes) {
                return null;
            }
            const { productId, itemCode, attributeId, attributeDetailId, attributeName, itemValue } = productAttributes;
            if (!productId || !attributeId || !attributeDetailId) {
                throw new Error('Missing required identifiers for product attribute upsert');
            }
            const filter = { productId, attributeId, attributeDetailId };
            const update = {
                $set: { productId, itemCode, attributeId, attributeDetailId, attributeName, itemValue, updatedAt: new Date() },
                $setOnInsert: { createdAt: new Date(), },
            };
            const options = { upsert: true, new: true, setDefaultsOnInsert: true, };
            const attributeValue = await this.productAttributesModel.findOneAndUpdate(filter, update, options).exec();
            return attributeValue;
        }
        catch (error) {
            console.error('Error in createProductAttributes:', error);
            return null;
        }
    }
    async deleteProductAttribute(productId) {
        try {
            await this.productAttributesModel.deleteMany({ productId }).lean();
            return true;
        }
        catch (error) {
            console.error('Error in deleteProductAttribute:', error);
            return false;
        }
    }
}
exports.default = ProductsInventoryService;
