"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const pagination_1 = require("../../../components/pagination");
const branch_config_1 = require("../../../utils/config/admin/operation/branch-config");
const items_config_1 = require("../../../utils/config/admin/inventory-management/items-config");
const job_order_config_1 = require("../../../utils/config/admin/job-order/job-order-config");
const job_order_master_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-master-model"));
const category_config_1 = require("../../../utils/config/admin/inventory-management/category-config");
const complaint_config_1 = require("../../../utils/config/admin/inventory-management/complaint-config");
const mongoose_1 = __importDefault(require("mongoose"));
const customer_config_1 = require("../../../utils/config/admin/customerer-managemet/customer-config");
const collections_1 = require("../../../constants/collections");
const lodash_1 = require("lodash");
const job_order_details_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-details-model"));
const job_order_used_items_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-used-items-model"));
const job_order_gallery_image_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-gallery-image-model"));
const job_order_assignment_history_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-assignment-history-model"));
const spare_requests_and_transfers_model_1 = __importDefault(require("../../../model/admin/inventory-management/items/spare-requests-and-transfers-model"));
const job_order_1 = require("../../../constants/admin/job-order");
const vendor_1 = require("../../../utils/config/outsourced/vendor");
const job_order_customer_item_history_model_1 = __importDefault(require("../../../model/admin/job-order/job-order-customer-item-history-model"));
const job_order_by_products_1 = __importDefault(require("../../../model/admin/job-order/job-order-by-products"));
class JobOrderService {
    async findAll(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        finalSort = defaultSort;
        let pipeline = [
            (0, customer_config_1.customerLookup)(),
            { $unwind: { path: "$customer", preserveNullAndEmptyArrays: true } },
            (0, branch_config_1.branchLookup)('createdBranchId'),
            {
                $lookup: {
                    from: `${collections_1.collections.jobOrder.jobOrderDetails}`,
                    localField: '_id',
                    foreignField: 'jobMasterId',
                    as: 'jobOrderDetail'
                }
            },
            { $unwind: { path: "$branch", preserveNullAndEmptyArrays: true } },
            { $match: query },
            (0, job_order_config_1.jobOrderProject)({ jobdetails: '1', getcustomer: '1', getbranch: '1' }),
            { $sort: finalSort },
            { $skip: skip },
            { $limit: limit },
        ];
        return job_order_master_model_1.default.aggregate(pipeline).exec();
    }
    async findAllJobOrderAssignmentHistory(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        let pipeline = [
            {
                $lookup: {
                    from: `${collections_1.collections.jobOrder.jobOrderMasters}`,
                    localField: 'jobMasterId',
                    foreignField: '_id',
                    as: 'jobOrder'
                }
            },
            {
                $lookup: {
                    from: `${collections_1.collections.jobOrder.jobOrderDetails}`,
                    localField: 'jobOrderDetailId',
                    foreignField: '_id',
                    as: 'jobOrderDetail'
                }
            },
            {
                $lookup: {
                    from: `${collections_1.collections.staffManagement.adminUsers}`,
                    localField: 'technicianId',
                    foreignField: '_id',
                    as: 'technician'
                }
            },
            { $unwind: { path: "$jobOrder", preserveNullAndEmptyArrays: true } },
            { $unwind: { path: "$jobOrderDetail", preserveNullAndEmptyArrays: true } },
            { $unwind: { path: "$technician", preserveNullAndEmptyArrays: true } },
            { $match: query },
            { $sort: finalSort },
            { $skip: skip },
            { $limit: limit },
            {
                $project: {
                    _id: 1,
                    taskId: 1,
                    jobMasterId: 1,
                    jobOrderDetailId: 1,
                    technicianId: 1,
                    reassignedFromId: 1,
                    branchId: 1,
                    assignedAt: 1,
                    taskTitle: 1,
                    reason: 1,
                    description: 1,
                    reassigned: 1,
                    priority: 1,
                    startDate: 1,
                    endDate: 1,
                    taskStartTime: 1,
                    taskEndTime: 1,
                    technicianStatus: 1,
                    technicianStatusAt: 1,
                    reassignedFromIdAt: 1,
                    createdBy: 1,
                    jobOrder: {
                        customerId: 1,
                        createdBranchId: 1,
                        jobId: 1,
                        jobTotalQuantity: 1,
                        jobSubTotalAmount: 1,
                        jobBalanceAmount: 1,
                        jobTotalAmount: 1,
                        costItems: 1,
                        jobOrderStatus: 1,
                        jobOrderStatusAt: 1,
                    },
                    jobOrderDetail: 1,
                    technician: 1,
                }
            }
        ];
        return job_order_assignment_history_model_1.default.aggregate(pipeline).exec();
    }
    async findAllJobOrderCustomerItemTransferHistory(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        let pipeline = [
            {
                $lookup: {
                    from: `${collections_1.collections.jobOrder.jobOrderMasters}`,
                    localField: 'jobOrderMasterId',
                    foreignField: '_id',
                    as: 'jobOrder'
                }
            },
            (0, customer_config_1.customerLookup)(),
            (0, branch_config_1.branchLookup)(),
            (0, branch_config_1.branchLookup)('transferFromBranchId', 'transferFromBranch'),
            {
                $lookup: {
                    from: `${collections_1.collections.staffManagement.adminUsers}`,
                    localField: 'driverId',
                    foreignField: '_id',
                    as: 'driver'
                }
            },
            { $unwind: { path: "$jobOrder", preserveNullAndEmptyArrays: true } },
            { $unwind: { path: "$customer", preserveNullAndEmptyArrays: true } },
            { $unwind: { path: "$branch", preserveNullAndEmptyArrays: true } },
            { $unwind: { path: "$transferFromBranch", preserveNullAndEmptyArrays: true } },
            { $unwind: { path: "$driver", preserveNullAndEmptyArrays: true } },
            { $match: query },
            { $sort: finalSort },
            {
                $project: {
                    _id: 1,
                    jobOrderMasterId: 1,
                    jobOrderDetailId: 1,
                    customerId: 1,
                    branchId: 1,
                    transferFromBranchId: 1,
                    driverId: 1,
                    serviceType: 1,
                    historyType: 1,
                    transferNotes: 1,
                    historyStatus: 1,
                    historyStatusAt: 1,
                    driverHistoryStatus: 1,
                    driverHistoryStatusAt: 1,
                    serviceStatusMessage: 1,
                    createdAt: 1,
                    jobOrder: {
                        _id: 1,
                        customerId: 1,
                        createdBranchId: 1,
                        jobId: 1,
                        jobTotalQuantity: 1,
                        jobSubTotalAmount: 1,
                        jobBalanceAmount: 1,
                        jobTotalAmount: 1,
                        costItems: 1,
                        jobOrderStatus: 1,
                        jobOrderStatusAt: 1,
                    },
                    customer: {
                        _id: 1,
                        customerTitleType: 1,
                        customerType: 1,
                        customerTitle: 1,
                        customerCode: 1,
                        customerEmail: 1,
                        customerPhoneNumber: 1,
                        customerCountryCode: 1,
                    },
                    branch: {
                        _id: 1,
                        branchTitle: 1,
                        branchCode: 1,
                    },
                    transferFromBranch: {
                        _id: 1,
                        branchTitle: 1,
                        branchCode: 1,
                    },
                    driver: {
                        _id: 1,
                        designationId: 1,
                        userTitle: 1,
                        userCode: 1,
                        userEmail: 1,
                        userPhoneNumber: 1,
                    }
                }
            }
        ];
        const facetPipeline = [
            {
                $facet: {
                    totalCount: [{ $match: query }, { $count: 'total' }],
                    data: [
                        { $skip: skip },
                        { $limit: limit }
                    ]
                }
            },
            {
                $project: {
                    totalCount: { $arrayElemAt: ['$totalCount.total', 0] },
                    data: 1
                }
            }
        ];
        pipeline.push(...facetPipeline);
        const retVal = await job_order_customer_item_history_model_1.default.aggregate(pipeline).exec();
        return {
            data: (0, lodash_1.get)(retVal, '[0].data', []),
            totalCount: (0, lodash_1.get)(retVal, '[0].totalCount', 0)
        };
    }
    async getTotalCount(query = {}) {
        try {
            const totalCount = await job_order_master_model_1.default.countDocuments(query);
            return totalCount;
        }
        catch (error) {
            throw new Error('Error fetching total count of jobOrder');
        }
    }
    async create(jobOrderData) {
        const createdJobOrder = await job_order_master_model_1.default.create(jobOrderData);
        if (createdJobOrder) {
            // const pipeline: any = [
            //     { $match: { _id: createdJobOrder._id } },
            //     itemMasterLookup,
            //     complaintLookup,
            //     categoryLookup,
            //     branchLookup,
            //     ...jobOrderProject,
            // ];
            // const [jobOrderData] = await JobOrderMasterModel.aggregate(pipeline).exec();
            return createdJobOrder;
        }
        else {
            return null;
        }
    }
    async findJobOrderDetails(options) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        let jobDetailsPipeline = [
            { $match: query },
            {
                $lookup: {
                    from: `${collections_1.collections.jobOrder.jobOrderMasters}`,
                    localField: 'jobMasterId',
                    foreignField: '_id',
                    as: 'jobOrder',
                    pipeline: [
                        (0, customer_config_1.customerLookup)(),
                        { $unwind: { path: "$customer", preserveNullAndEmptyArrays: true } },
                    ]
                }
            },
            { $unwind: { path: "$jobOrder", preserveNullAndEmptyArrays: true } },
            complaint_config_1.complaintLookup,
            (0, branch_config_1.branchLookup)(),
            { $unwind: { path: "$complaint", preserveNullAndEmptyArrays: true } },
            { $unwind: { path: "$branch", preserveNullAndEmptyArrays: true } },
            { $match: query },
            { $skip: skip },
            { $limit: limit },
            {
                $lookup: {
                    from: `${collections_1.collections.inventoryManagement.items.itemsMaster}`,
                    localField: 'itemMasterId',
                    foreignField: '_id',
                    as: 'itemMaster'
                }
            },
            { $unwind: { path: "$itemMaster", preserveNullAndEmptyArrays: true } },
            category_config_1.categoryLookup,
            { $unwind: { path: "$category", preserveNullAndEmptyArrays: true } },
            vendor_1.vendorLookup,
            { $unwind: { path: "$vendor", preserveNullAndEmptyArrays: true } },
            { $sort: finalSort },
            (0, job_order_config_1.jobOrderDetailsProject)({ getjobOrder: '1', getcategory: '1', getcomplaint: '1', getbranch: '1', getvendor: "1", getgalleryimages: '1' })
        ];
        const facetPipeline = [
            {
                $facet: {
                    totalCount: [{ $match: query }, { $count: 'total' }],
                    data: [
                        { $skip: skip },
                        { $limit: limit }
                    ]
                }
            },
            {
                $project: {
                    totalCount: { $arrayElemAt: ['$totalCount.total', 0] },
                    data: 1
                }
            }
        ];
        jobDetailsPipeline.push(...facetPipeline);
        return await job_order_details_model_1.default.aggregate(jobDetailsPipeline).exec();
    }
    async findAllRequestedItems(options) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        let itemRequesPipeline = [
            { $match: query },
            { $match: query },
            {
                $lookup: {
                    from: `${collections_1.collections.operations.branches}`,
                    localField: 'currentBranchId',
                    foreignField: '_id',
                    as: 'requestedBranch'
                }
            },
            {
                $lookup: {
                    from: `${collections_1.collections.inventoryManagement.items.itemsMaster}`,
                    localField: 'itemMasterId',
                    foreignField: '_id',
                    as: 'itemMaster'
                }
            },
            {
                $lookup: {
                    from: `${collections_1.collections.inventoryManagement.items.itemsDetails}`,
                    localField: 'itemId',
                    foreignField: '_id',
                    as: 'itemDetail'
                }
            },
            {
                $lookup: {
                    from: `${collections_1.collections.inventoryManagement.items.stockInventory}`,
                    let: { itemId: '$itemId', branchId: '$newBranchId' }, // current branch
                    pipeline: [
                        {
                            $match: {
                                $expr: {
                                    $and: [
                                        { $eq: ['$itemId', '$$itemId'] },
                                        { $eq: ['$branchId', '$$branchId'] }
                                    ]
                                }
                            }
                        }
                    ],
                    as: 'stockDetails'
                }
            },
            {
                $lookup: {
                    from: `${collections_1.collections.inventoryManagement.items.serialNumbers}`,
                    let: { itemId: '$itemId', branchId: '$newBranchId' }, // current branch
                    pipeline: [
                        {
                            $match: {
                                $expr: {
                                    $and: [
                                        { $eq: ['$itemId', '$$itemId'] },
                                        { $eq: ['$branchId', '$$branchId'] }
                                    ]
                                },
                                stockType: job_order_1.stockTypes.in
                            }
                        }
                    ],
                    as: 'stockWithSerialDetails'
                }
            },
            { $unwind: { path: "$requestedBranch", preserveNullAndEmptyArrays: true } },
            { $unwind: { path: "$itemMaster", preserveNullAndEmptyArrays: true } },
            { $unwind: { path: "$itemDetail", preserveNullAndEmptyArrays: true } },
            { $unwind: { path: "$stockDetails", preserveNullAndEmptyArrays: true } },
            {
                $project: {
                    _id: 1,
                    type: 1,
                    transferredBy: 1,
                    transferredAt: 1,
                    requestedStatusAt: 1,
                    requestedStatus: 1,
                    requestedBy: 1,
                    requestedAt: 1,
                    remarks: 1,
                    reason: 1,
                    quantity: 1,
                    priority: 1,
                    jobOrderDetailId: 1,
                    jobMasterId: 1,
                    currentBranchId: 1,
                    createdAt: 1,
                    itemDetail: {
                        _id: 1,
                        itemCode: 1,
                        itemType: 1,
                        itemPrice: 1,
                        itemTypeQuantity: 1,
                        itemMasterId: 1,
                        itemImageUrl: 1,
                        barCode: 1,
                        itemTitle: 1,
                        totalItemStock: 1,
                    },
                    itemMaster: {
                        _id: 1,
                        categoryId: 1,
                        itemMasterCode: 1,
                        itemMasterTitle: 1,
                        bomMasterQuantity: 1,
                        bomMasterOldQuantity: 1,
                        itemMasterPrice: 1,
                        slug: 1,
                    },
                    requestedBranch: {
                        _id: 1,
                        branchTitle: 1,
                        branchCode: 1,
                    },
                    stockDetails: {
                        _id: 1,
                        itemStock: 1,
                        branchId: 1,
                        itemId: 1,
                        itemMasterId: 1,
                    },
                    stockWithSerialDetails: {
                        _id: 1,
                        branchId: 1,
                        itemId: 1,
                        itemMasterId: 1,
                        serialNumber: 1,
                        serialNumberType: 1,
                        stockCount: 1,
                        stockType: 1,
                    }
                }
            },
            { $sort: finalSort },
        ];
        const facetPipeline = [
            {
                $facet: {
                    totalCount: [{ $match: query }, { $count: 'total' }],
                    data: [
                        { $skip: skip },
                        { $limit: limit }
                    ]
                }
            },
            {
                $project: {
                    totalCount: { $arrayElemAt: ['$totalCount.total', 0] },
                    data: 1
                }
            }
        ];
        itemRequesPipeline.push(...facetPipeline);
        return await spare_requests_and_transfers_model_1.default.aggregate(itemRequesPipeline).exec();
    }
    async findOneSimple(options) {
        const { jobOrderId, simple = '1', jobdetails = '0', getcustomer = '0', getbranch = '1', gettechnician = '0', jobDetailQuery, contextTechnicianId } = options;
        let pipeline = [];
        const addCustomerLookup = () => {
            pipeline.push((0, customer_config_1.customerLookup)());
            pipeline.push({ $unwind: { path: "$customer", preserveNullAndEmptyArrays: true } });
        };
        const addBranchLookup = () => {
            pipeline.push((0, branch_config_1.branchLookup)('createdBranchId'));
            pipeline.push({ $unwind: { path: "$branch", preserveNullAndEmptyArrays: true } });
        };
        if (simple === '1') {
            if (getcustomer === '1')
                addCustomerLookup();
            pipeline.push((0, job_order_config_1.jobOrderProject)({ simple, getcustomer }));
        }
        else {
            if (getcustomer === '1')
                addCustomerLookup();
            if (getbranch === '1')
                addBranchLookup();
            pipeline.push((0, job_order_config_1.jobOrderProject)({ simple, jobdetails, getcustomer, getbranch }));
        }
        const jobOrderMasterPipeline = [
            { $match: { _id: new mongoose_1.default.Types.ObjectId(jobOrderId) } },
            ...pipeline,
        ];
        let [jobOrderData] = await job_order_master_model_1.default.aggregate(jobOrderMasterPipeline).exec();
        let assignmentHistoryPipeline = [
            {
                $match: {
                    ...(contextTechnicianId
                        ? {
                            $or: [
                                { technicianId: new mongoose_1.default.Types.ObjectId(contextTechnicianId) },
                                { reassignedFromId: new mongoose_1.default.Types.ObjectId(contextTechnicianId) },
                            ],
                        }
                        : {}),
                    jobMasterId: new mongoose_1.default.Types.ObjectId(jobOrderId),
                },
            },
            {
                $lookup: {
                    from: `${collections_1.collections.staffManagement.adminUsers}`,
                    localField: 'technicianId',
                    foreignField: '_id',
                    as: 'technician',
                },
            },
            { $unwind: { path: "$technician", preserveNullAndEmptyArrays: true } },
        ];
        const jobOrderAssignmentHistory = await job_order_assignment_history_model_1.default.aggregate(assignmentHistoryPipeline).exec();
        if (jobdetails === '1' && !(0, lodash_1.isEmpty)(jobOrderData)) {
            const jobOrderDetailIds = jobOrderAssignmentHistory.map((history) => history.jobOrderDetailId);
            let jobDetailsPipeline = [
                {
                    $match: {
                        ...(contextTechnicianId && jobOrderDetailIds.length > 0 ? {
                            $or: [
                                { _id: { $in: jobOrderDetailIds } },
                            ],
                        } : jobDetailQuery ? { ...jobDetailQuery } : {}),
                        jobMasterId: new mongoose_1.default.Types.ObjectId(jobOrderId),
                    },
                },
                {
                    $lookup: {
                        from: `${collections_1.collections.inventoryManagement.items.itemsMaster}`,
                        localField: 'itemMasterId',
                        foreignField: '_id',
                        as: 'itemMaster'
                    }
                },
                { $unwind: { path: "$itemMaster", preserveNullAndEmptyArrays: true } },
                category_config_1.categoryLookup,
                complaint_config_1.complaintLookup,
                (0, branch_config_1.branchLookup)(),
                (0, branch_config_1.branchLookup)('transferBranchId', 'transferBranch'),
                vendor_1.vendorLookup,
                { $unwind: { path: "$category", preserveNullAndEmptyArrays: true } },
                { $unwind: { path: "$complaint", preserveNullAndEmptyArrays: true } },
                { $unwind: { path: "$branch", preserveNullAndEmptyArrays: true } },
                { $unwind: { path: "$transferBranch", preserveNullAndEmptyArrays: true } },
                { $unwind: { path: "$vendor", preserveNullAndEmptyArrays: true } },
                (0, job_order_config_1.jobOrderDetailsProject)({ getcategory: '1', getcomplaint: '1', getbranch: '1', gettransferbranch: '1', getgalleryimages: '1', getvendor: "1", gettechnician })
            ];
            let jobOrderDetails = await job_order_details_model_1.default.aggregate(jobDetailsPipeline).exec();
            if (!(0, lodash_1.isEmpty)(jobOrderDetails)) {
                const itemsPipeline = [
                    { $match: { jobMasterId: new mongoose_1.default.Types.ObjectId(jobOrderId) } },
                    {
                        $lookup: {
                            from: `${collections_1.collections.inventoryManagement.items.itemsDetails}`,
                            localField: 'itemId',
                            foreignField: '_id',
                            as: 'itemDetail'
                        }
                    },
                    {
                        $lookup: {
                            from: `${collections_1.collections.staffManagement.adminUsers}`,
                            localField: 'technicianId',
                            foreignField: '_id',
                            as: 'technician'
                        }
                    },
                    { $unwind: { path: "$technician", preserveNullAndEmptyArrays: true } },
                    { $unwind: { path: "$itemDetail", preserveNullAndEmptyArrays: true } },
                    // { $match: { technicianId: { $exists: true, $ne: null } } },
                    {
                        $project: {
                            itemId: 1,
                            jobOrderDetailId: 1,
                            itemMasterId: 1,
                            jobMasterId: 1,
                            technicianId: 1,
                            itemType: 1,
                            quantity: 1,
                            buyedItemPrice: 1,
                            itemPrice: 1,
                            itemTypeQuantity: 1,
                            createdBy: 1,
                            createdAt: 1,
                            itemDetail: {
                                _id: 1,
                                itemCode: 1,
                                itemType: 1,
                                itemTitle: 1,
                                itemPrice: 1,
                                itemTypeQuantity: 1,
                                totalItemStock: 1,
                                slug: 1,
                                itemImageUrl: 1,
                            },
                            technician: {
                                _id: 1,
                                designationId: 1,
                                userTitle: 1,
                                userCode: 1,
                                userEmail: 1,
                                userPhoneNumber: 1,
                            }
                        }
                    }
                ];
                const itemDetails = await job_order_used_items_model_1.default.aggregate(itemsPipeline).exec();
                if ((0, lodash_1.size)(jobOrderDetails) > 0 && (0, lodash_1.size)(itemDetails) > 0) {
                    jobOrderDetails = jobOrderDetails.map((jobDetail) => {
                        const matchingItems = itemDetails.filter((item) => item.jobOrderDetailId?.toString() === jobDetail._id?.toString());
                        return {
                            ...jobDetail,
                            items: matchingItems || []
                        };
                    });
                }
            }
            const jobOrderByProducts = await job_order_by_products_1.default.find({ jobMasterId: new mongoose_1.default.Types.ObjectId(jobOrderId) });
            const jobOrderGalleryImages = await job_order_gallery_image_model_1.default.find({ jobOrderMasterId: new mongoose_1.default.Types.ObjectId(jobOrderId) });
            return {
                ...jobOrderData,
                jobOrderDetails,
                jobOrderAssignmentHistory,
                jobOrderGalleryImages,
                jobOrderByProducts
            };
        }
        return jobOrderData || null;
    }
    async update(jobOrderId, jobOrderData) {
        const updatdJobOrder = await job_order_master_model_1.default.findByIdAndUpdate(jobOrderId, jobOrderData, { new: true, useFindAndModify: false });
        if (updatdJobOrder) {
            const pipeline = [
                { $match: { _id: new mongoose_1.default.Types.ObjectId(jobOrderId) } },
                items_config_1.itemMasterLookup,
                complaint_config_1.complaintLookup,
                category_config_1.categoryLookup,
                (0, branch_config_1.branchLookup)(),
                job_order_config_1.jobOrderProject,
            ];
            const [jobOrderData] = await job_order_master_model_1.default.aggregate(pipeline).exec();
            return jobOrderData;
        }
        else {
            return null;
        }
    }
    async destroy(jobOrderId) {
        return job_order_master_model_1.default.findOneAndDelete({ _id: jobOrderId });
    }
}
exports.default = new JobOrderService();
