"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const mongoose_1 = __importDefault(require("mongoose"));
const pagination_1 = require("../../../components/pagination");
const collections_1 = require("../../../constants/collections");
const branch_model_1 = __importDefault(require("../../../model/admin/operations/branch-model"));
const branch_vistor_id_model_1 = __importDefault(require("../../../model/admin/operations/branch-vistor-id-model"));
const lodash_1 = require("lodash");
class BranchService {
    async findAll(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const { getItemStock, currentBranchId } = options;
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        const pipeline = [];
        if (getItemStock === '1') {
            pipeline.push({
                $lookup: {
                    from: collections_1.collections.inventoryManagement.items.stockInventory,
                    localField: '_id',
                    foreignField: 'branchId',
                    as: 'stockInventory',
                    ...(query['stockInventory.itemId'] ? {
                        pipeline: [
                            {
                                $match: { itemId: query['stockInventory.itemId'] }
                            }
                        ]
                    } : {})
                }
            }, { $unwind: { path: '$stockInventory', preserveNullAndEmptyArrays: true } }, {
                $group: {
                    _id: '$_id',
                    branchTitle: { $first: '$branchTitle' },
                    branchCode: { $first: '$branchCode' },
                    slug: { $first: '$slug' },
                    status: { $first: '$status' },
                    createdBy: { $first: '$createdBy' },
                    createdAt: { $first: '$createdAt' },
                    stockInventory: { $push: '$stockInventory' }
                }
            }, {
                $project: {
                    _id: 1,
                    branchTitle: 1,
                    branchCode: 1,
                    slug: 1,
                    status: 1,
                    createdBy: 1,
                    createdAt: 1,
                    stockInventory: 1,
                }
            });
        }
        const facetPipeline = {
            totalCount: [{ $count: "count" }],
            paginatedResults: [
                ...((0, lodash_1.size)(finalSort) ? [{ $sort: finalSort }] : []),
                ...(skip !== undefined ? [{ $skip: skip }] : []),
                ...(limit !== undefined ? [{ $limit: limit }] : [])
            ]
        };
        if (Object.keys(query).length) {
            pipeline.push({ $match: query });
        }
        pipeline.push({ $facet: facetPipeline });
        const result = await branch_model_1.default.aggregate(pipeline).exec();
        let totalCount = 0;
        let paginatedResults = [];
        let currentBranch = null;
        if ((0, lodash_1.size)(result) && (0, lodash_1.size)((0, lodash_1.get)(result, '[0].paginatedResults'))) {
            totalCount = (0, lodash_1.get)(result, '[0].totalCount[0].count', 0);
            paginatedResults = (0, lodash_1.get)(result, '[0].paginatedResults', []);
        }
        if (currentBranchId) {
            const currentBranchResult = await branch_model_1.default.findOne({ _id: new mongoose_1.default.Types.ObjectId(currentBranchId) }).exec();
            currentBranch = currentBranchResult ? currentBranchResult : null;
        }
        return {
            totalCount,
            paginatedResults,
            currentBranch
        };
    }
    async findAllBranchVisitors(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        const pipeline = [
            { $match: query },
            {
                $lookup: {
                    from: collections_1.collections.operations.branches,
                    localField: 'branchId',
                    foreignField: '_id',
                    as: 'visitorBranch'
                }
            },
            { $unwind: { path: '$visitorBranch', preserveNullAndEmptyArrays: true } }
        ];
        if (skip !== undefined)
            pipeline.push({ $skip: skip });
        if (limit !== undefined)
            pipeline.push({ $limit: limit });
        if (finalSort)
            pipeline.push({ $sort: finalSort });
        return await branch_vistor_id_model_1.default.aggregate(pipeline).exec();
    }
    async getTotalCount(query = {}) {
        try {
            const totalCount = await branch_model_1.default.countDocuments(query);
            return totalCount;
        }
        catch (error) {
            throw new Error('Error fetching total count of branch');
        }
    }
    async create(branchData) {
        return branch_model_1.default.create(branchData);
    }
    async findOne(branchId) {
        return branch_model_1.default.findById(branchId);
    }
    async update(branchId, branchData) {
        return branch_model_1.default.findByIdAndUpdate(branchId, branchData, { new: true, useFindAndModify: false });
    }
    async destroy(branchId) {
        return branch_model_1.default.findOneAndDelete({ _id: branchId });
    }
}
exports.default = new BranchService();
