"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const pagination_1 = require("../../../components/pagination");
const branch_config_1 = require("../../../utils/config/admin/operation/branch-config");
const admin_user_config_1 = require("../../../utils/config/admin/staff-management/admin-user-config");
const department_config_1 = require("../../../utils/config/admin/staff-management/department-config");
const designation_config_1 = require("../../../utils/config/admin/staff-management/designation-config");
const general_config_1 = require("../../../utils/config/general/general-config");
const admin_user_model_1 = __importDefault(require("../../../model/admin/staff-management/admin-user-model"));
const lodash_1 = require("lodash");
class AdminUserinventory {
    async findAll(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        const pipeline = [
            { $sort: finalSort },
        ];
        const facetPipeline = [
            {
                $facet: {
                    totalCount: [
                        { $count: 'total' }
                    ],
                    data: [
                        { $skip: skip },
                        { $limit: limit },
                        ...((query['designation.slug'] === '' || query['designation.slug'] === undefined) ? [
                            designation_config_1.designationLookup,
                        ] : []),
                        (0, branch_config_1.branchLookup)(),
                        department_config_1.departmentLookup,
                        general_config_1.countriesLookup,
                        general_config_1.statesLookup,
                        general_config_1.citiesLookup,
                        ...admin_user_config_1.userProject
                    ]
                }
            },
            {
                $project: {
                    totalCount: { $ifNull: [{ $arrayElemAt: ['$totalCount.total', 0] }, 0] },
                    data: 1
                }
            }
        ];
        if (query['designation.slug']) {
            pipeline.push(designation_config_1.designationLookup);
        }
        pipeline.push({ $match: query });
        pipeline.push(...facetPipeline);
        const retVal = await admin_user_model_1.default.aggregate(pipeline).exec();
        return {
            data: (0, lodash_1.get)(retVal, '[0].data', []),
            totalCount: (0, lodash_1.get)(retVal, '[0].totalCount', 0)
        };
    }
    async getTotalCount(query = {}) {
        try {
            const totalCount = await admin_user_model_1.default.countDocuments(query);
            return totalCount;
        }
        catch (error) {
            throw new Error('Error fetching total count of users');
        }
    }
    async create(userData) {
        const createdUser = await admin_user_model_1.default.create(userData);
        if (createdUser) {
            const pipeline = [
                { $match: { _id: createdUser._id } },
                designation_config_1.designationLookup,
                (0, branch_config_1.branchLookup)(),
                department_config_1.departmentLookup,
                general_config_1.countriesLookup,
                general_config_1.statesLookup,
                general_config_1.citiesLookup,
                ...admin_user_config_1.userProject,
            ];
            const [userWithBranch] = await admin_user_model_1.default.aggregate(pipeline).exec();
            return userWithBranch;
        }
        else {
            return null;
        }
    }
    async findOne(userId) {
        const pipeline = [
            { $match: { _id: userId } },
            designation_config_1.designationLookup,
            (0, branch_config_1.branchLookup)(),
            department_config_1.departmentLookup,
            general_config_1.countriesLookup,
            general_config_1.statesLookup,
            general_config_1.citiesLookup,
            ...admin_user_config_1.userProject,
        ];
        const [userWithDetails] = await admin_user_model_1.default.aggregate(pipeline).exec();
        return userWithDetails || null;
    }
    async update(userId, userData) {
        const updatdUser = await admin_user_model_1.default.findByIdAndUpdate(userId, userData, { new: true, useFindAndModify: false });
        if (updatdUser) {
            const pipeline = [
                { $match: { _id: userId } },
                designation_config_1.designationLookup,
                (0, branch_config_1.branchLookup)(),
                department_config_1.departmentLookup,
                general_config_1.countriesLookup,
                general_config_1.statesLookup,
                general_config_1.citiesLookup,
                ...admin_user_config_1.userProject,
            ];
            const [userWithBranch] = await admin_user_model_1.default.aggregate(pipeline).exec();
            return userWithBranch;
        }
        else {
            return null;
        }
    }
    async destroy(userId) {
        return admin_user_model_1.default.findOneAndDelete({ _id: userId });
    }
}
exports.default = new AdminUserinventory();
