"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const lodash_1 = require("lodash");
const pagination_1 = require("../../components/pagination");
const conditions_model_1 = __importDefault(require("../../model/admin/other/conditions-model"));
const category_config_1 = require("../../utils/config/admin/inventory-management/category-config");
const condition_config_1 = require("../../utils/config/admin/other/condition-config");
class ConditionService {
    async findAll(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        let pipeline = [
            { $match: query },
            { $sort: finalSort },
            category_config_1.categoryLookup,
            ...condition_config_1.conditionProject,
        ];
        const facetPipeline = [
            {
                $facet: {
                    totalCount: [
                        { $count: 'total' }
                    ],
                    data: [
                        { $skip: skip },
                        { $limit: limit }
                    ]
                }
            },
            {
                $project: {
                    totalCount: { $ifNull: [{ $arrayElemAt: ['$totalCount.total', 0] }, 0] },
                    data: 1
                }
            }
        ];
        pipeline.push(...facetPipeline);
        const retVal = await conditions_model_1.default.aggregate(pipeline).exec();
        return {
            data: (0, lodash_1.get)(retVal, '[0].data', []),
            totalCount: (0, lodash_1.get)(retVal, '[0].totalCount', 0)
        };
    }
    async getTotalCount(query = {}) {
        try {
            const totalCount = await conditions_model_1.default.countDocuments(query);
            return totalCount;
        }
        catch (error) {
            throw new Error('Error fetching total count of condition');
        }
    }
    async create(conditionData) {
        const createdCondition = await conditions_model_1.default.create(conditionData);
        if (createdCondition) {
            const pipeline = [
                { $match: { _id: createdCondition._id } },
                category_config_1.categoryLookup,
                ...condition_config_1.conditionProject,
            ];
            const [conditionWithBranch] = await conditions_model_1.default.aggregate(pipeline).exec();
            return conditionWithBranch;
        }
        else {
            return null;
        }
    }
    async findOne(conditionId) {
        const pipeline = [
            { $match: { _id: conditionId } },
            category_config_1.categoryLookup,
            ...condition_config_1.conditionProject,
        ];
        const [conditionWithBranch] = await conditions_model_1.default.aggregate(pipeline).exec();
        return conditionWithBranch;
    }
    async update(conditionId, conditionData) {
        const updatdCondition = await conditions_model_1.default.findByIdAndUpdate(conditionId, conditionData, { new: true, useFindAndModify: false });
        if (updatdCondition) {
            const pipeline = [
                { $match: { _id: updatdCondition._id } },
                category_config_1.categoryLookup,
                ...condition_config_1.conditionProject,
            ];
            const [conditionWithBranch] = await conditions_model_1.default.aggregate(pipeline).exec();
            return conditionWithBranch;
        }
        else {
            return null;
        }
    }
    async destroy(conditionId) {
        return conditions_model_1.default.findOneAndDelete({ _id: conditionId });
    }
}
exports.default = new ConditionService();
