"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const lodash_1 = require("lodash");
const pagination_1 = require("../../components/pagination");
const service_status_model_1 = __importDefault(require("../../model/admin/other/service-status-model"));
const category_config_1 = require("../../utils/config/admin/inventory-management/category-config");
const service_status_config_1 = require("../../utils/config/admin/other/service-status-config");
class ServiceStatusService {
    async findAll(options = {}) {
        const { query, skip, limit, sort } = (0, pagination_1.pagination)(options.query || {}, options);
        const defaultSort = { createdAt: -1 };
        let finalSort = sort || defaultSort;
        const sortKeys = Object.keys(finalSort);
        if (sortKeys.length === 0) {
            finalSort = defaultSort;
        }
        let pipeline = [
            { $match: query },
            { $sort: finalSort },
            category_config_1.categoryLookup,
            ...service_status_config_1.serviceStatusProject,
        ];
        const facetPipeline = [
            {
                $facet: {
                    totalCount: [
                        { $count: 'total' }
                    ],
                    data: [
                        { $skip: skip },
                        { $limit: limit }
                    ]
                }
            },
            {
                $project: {
                    totalCount: { $ifNull: [{ $arrayElemAt: ['$totalCount.total', 0] }, 0] },
                    data: 1
                }
            }
        ];
        pipeline.push(...facetPipeline);
        const retVal = await service_status_model_1.default.aggregate(pipeline).exec();
        return {
            data: (0, lodash_1.get)(retVal, '[0].data', []),
            totalCount: (0, lodash_1.get)(retVal, '[0].totalCount', 0)
        };
    }
    async getTotalCount(query = {}) {
        try {
            const totalCount = await service_status_model_1.default.countDocuments(query);
            return totalCount;
        }
        catch (error) {
            throw new Error('Error fetching total count of service status');
        }
    }
    async create(serviceStatusData) {
        const createdServiceStatus = await service_status_model_1.default.create(serviceStatusData);
        if (createdServiceStatus) {
            const pipeline = [
                { $match: { _id: createdServiceStatus._id } },
                category_config_1.categoryLookup,
                ...service_status_config_1.serviceStatusProject,
            ];
            const [serviceStatusWithBranch] = await service_status_model_1.default.aggregate(pipeline).exec();
            return serviceStatusWithBranch;
        }
        else {
            return null;
        }
    }
    async findOne(serviceStatusId) {
        const pipeline = [
            { $match: { _id: serviceStatusId } },
            category_config_1.categoryLookup,
            ...service_status_config_1.serviceStatusProject,
        ];
        const [serviceStatusWithBranch] = await service_status_model_1.default.aggregate(pipeline).exec();
        return serviceStatusWithBranch;
    }
    async update(serviceStatusId, serviceStatusData) {
        const updatdServiceStatus = await service_status_model_1.default.findByIdAndUpdate(serviceStatusId, serviceStatusData, { new: true, useFindAndModify: false });
        if (updatdServiceStatus) {
            const pipeline = [
                { $match: { _id: updatdServiceStatus._id } },
                category_config_1.categoryLookup,
                ...service_status_config_1.serviceStatusProject,
            ];
            const [serviceStatusWithBranch] = await service_status_model_1.default.aggregate(pipeline).exec();
            return serviceStatusWithBranch;
        }
        else {
            return null;
        }
    }
    async destroy(serviceStatusId) {
        return service_status_model_1.default.findOneAndDelete({ _id: serviceStatusId });
    }
}
exports.default = new ServiceStatusService();
