"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.itemRequestAcceptFormSchema = exports.itemStockUpdateSchema = exports.itemDetailStatusSchema = exports.itemMasterStatusSchema = exports.itemDetailSchema = exports.updateBomItemQuantitySchema = exports.itemMasterSchema = void 0;
const mongoose_1 = __importDefault(require("mongoose"));
const zod_1 = require("zod");
const collections_1 = require("../../../../../constants/collections");
const item_details_serial_number_1 = __importDefault(require("../../../../../model/admin/inventory-management/items/item-details-serial-number"));
const lodash_1 = require("lodash");
const items_1 = require("../../../../../constants/items");
const itemMasterAttributeSchema = zod_1.z.object({
    attributeId: zod_1.z.string({ required_error: 'Key column is required', }).min(1, { message: "Attribute ID is required" }),
    attributeDetailId: zod_1.z.string({ required_error: 'Key column is required', }).min(1, { message: "Attribute Detail ID is required" }),
});
exports.itemMasterSchema = zod_1.z.array(zod_1.z.object({
    _id: zod_1.z.string().optional(),
    categoryId: zod_1.z.string().refine((id) => mongoose_1.default.Types.ObjectId.isValid(id), {
        message: 'Invalid category ID',
    }),
    itemMasterTitle: zod_1.z.string().min(1, { message: 'Item title is required.' }),
    itemMasterCode: zod_1.z.string().min(3, { message: 'Item master code must be at least 3 characters long.' }),
    itemType: zod_1.z.enum([items_1.itemTypes.normal, items_1.itemTypes.bom]).optional(),
    itemMasterDescription: zod_1.z.string().optional(),
    itemMasterImageUrl: zod_1.z.string().url({ message: 'Item image URL must be a valid URL.' }).optional(),
    itemMasterAttributes: zod_1.z.any(),
    status: zod_1.z.enum(['1', '2', '3']).optional(),
})).superRefine(async (dataArray, ctx) => {
    const itemMasterCodeSet = new Set();
    const itemMasterCodes = dataArray.map(data => data.itemMasterCode);
    const existingItems = await mongoose_1.default.model(collections_1.collections.inventoryManagement.items.itemsMaster)
        .find({ itemMasterCode: { $in: itemMasterCodes } })
        .select('itemMasterCode _id')
        .lean();
    const existingCodeMap = existingItems.reduce((acc, item) => {
        acc[item.itemMasterCode] = item._id.toString();
        return acc;
    }, {});
    for (const [index, data] of dataArray.entries()) {
        const { _id, itemMasterCode, itemMasterAttributes } = data;
        const id = _id ? new mongoose_1.default.Types.ObjectId(_id) : undefined;
        if (itemMasterCodeSet.has(itemMasterCode)) {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                path: [index, 'itemMasterCode'],
                message: `Item master code '${itemMasterCode}' is duplicated at index ${index}.`,
            });
        }
        else {
            itemMasterCodeSet.add(itemMasterCode);
        }
        const existingItemId = existingCodeMap[itemMasterCode];
        if (existingItemId && (!id || existingItemId !== id.toString())) {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                path: [index, 'itemMasterCode'],
                message: 'Item master code must be unique.',
            });
        }
        if (!Array.isArray(itemMasterAttributes)) {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                path: [index, 'itemMasterAttributes'],
                message: 'itemMasterAttributes must be an array of attributes',
            });
            continue;
        }
        const attributeIds = new Set();
        for (let attrIndex = 0; attrIndex < itemMasterAttributes.length; attrIndex++) {
            const attribute = itemMasterAttributes[attrIndex];
            const result = itemMasterAttributeSchema.safeParse(attribute);
            if (!result.success) {
                result.error.issues.forEach(issue => {
                    ctx.addIssue({
                        code: issue.code,
                        path: [index, 'itemMasterAttributes', attrIndex, ...issue.path],
                        message: issue.message,
                    });
                });
            }
            else {
                if (attributeIds.has(attribute.attributeDetailId)) {
                    ctx.addIssue({
                        code: zod_1.z.ZodIssueCode.custom,
                        path: [index, 'itemMasterAttributes', attrIndex, 'attributeDetailId'],
                        message: `Duplicate attributeDetailId '${attribute.attributeDetailId}' found at index ${index} for attribute index ${attrIndex}.`,
                    });
                }
                else {
                    attributeIds.add(attribute.attributeDetailId);
                }
            }
        }
    }
});
exports.updateBomItemQuantitySchema = zod_1.z
    .array(zod_1.z.object({
    itemMasterCode: zod_1.z.string().min(3, { message: 'Item master code must be at least 3 characters long.' }),
    itemMasterPrice: zod_1.z.number().optional(),
    bomMasterQuantity: zod_1.z.number().optional(),
}))
    .min(1, { message: 'At least one item is required.' })
    .superRefine((dataArray, ctx) => {
    dataArray.forEach((item, index) => {
        if (!(item.itemMasterPrice && item.itemMasterPrice > 0) &&
            !(item.bomMasterQuantity && item.bomMasterQuantity > 0)) {
            ctx.addIssue({
                code: 'custom',
                message: 'At least one of itemMasterPrice or bomMasterQuantity must be greater than 0.',
                path: [index, 'bomMasterQuantity'],
            });
        }
    });
});
exports.itemDetailSchema = zod_1.z.array(zod_1.z.object({
    _id: zod_1.z.string().optional(),
    itemMasterId: zod_1.z.string()
        .refine((id) => mongoose_1.default.Types.ObjectId.isValid(id), {
        message: 'Invalid item master ID',
    }).refine(async (id) => {
        const exists = await mongoose_1.default.model(collections_1.collections.inventoryManagement.items.itemsMaster).countDocuments({ _id: id });
        return exists > 0;
    }, {
        message: 'Item Master ID does not exist in the database.',
    }),
    itemType: zod_1.z.enum([items_1.itemTypes.normal, items_1.itemTypes.bom]).optional(),
    bomMasterQuantity: zod_1.z.number().optional(),
    itemMasterPrice: zod_1.z.number().min(0, { message: 'Item price must be a positive number.' }).optional(),
    items: zod_1.z.array(zod_1.z.object({
        itemId: zod_1.z.string().optional(),
        warrantyId: zod_1.z.string().optional().nullable(),
        itemTitle: zod_1.z.string().min(1, { message: 'Item title is required.' }),
        itemCode: zod_1.z.string().min(3, { message: 'Item code must be at least 3 characters long.' }),
        itemTypeQuantity: zod_1.z.number().optional(),
        itemPrice: zod_1.z.number().min(0, { message: 'Item price must be a positive number.' }).optional(),
        totalItemStock: zod_1.z.number().min(0, { message: 'Total item stock must be a positive number.' }).optional(),
        barCode: zod_1.z.string().optional(),
        hsnCode: zod_1.z.string().optional(),
        unit: zod_1.z.string().optional(),
        measurements: zod_1.z.object({
            weight: zod_1.z.string().optional(),
            height: zod_1.z.string().optional(),
            length: zod_1.z.string().optional(),
            width: zod_1.z.string().optional(),
        }).optional(),
        itemDescription: zod_1.z.string().optional(),
        itemImageUrl: zod_1.z.string().url({ message: 'Item image URL must be a valid URL.' }).optional(),
        qualityType: zod_1.z.string().optional(),
        stockInventory: zod_1.z.array(zod_1.z.object({
            branchCode: zod_1.z.string().refine(async (branchCode) => {
                const exists = await mongoose_1.default.model(collections_1.collections.operations.branches).countDocuments({ branchCode });
                return exists > 0;
            }, {
                message: 'Branch code does not exist in the database.',
            }),
            itemStock: zod_1.z.number().min(0, { message: 'Item stock must be a positive number.' }),
        })).optional(),
        status: zod_1.z.enum(['1', '2', '3']).default('1'),
    }))
})).superRefine(async (itemsArray, ctx) => {
    for (const [index, item] of itemsArray.entries()) {
        const { items } = item;
        if (Array.isArray(items)) {
            for (const [itemIndex, nestedItem] of items.entries()) {
                const { itemId, itemCode, stockInventory } = nestedItem;
                const itemDetailId = itemId ? new mongoose_1.default.Types.ObjectId(itemId) : undefined;
                const codeExists = await mongoose_1.default.model(collections_1.collections.inventoryManagement.items.itemsDetails).countDocuments({
                    itemCode, _id: { $ne: itemDetailId },
                    itemType: items_1.itemTypes.normal
                });
                if (codeExists > 0) {
                    ctx.addIssue({
                        code: zod_1.z.ZodIssueCode.custom,
                        path: [index, 'items', itemIndex, 'itemCode'],
                        message: 'Item code must be unique.',
                    });
                }
                if (stockInventory && stockInventory.length > 0) {
                    for (const [inventoryIndex, inventory] of stockInventory.entries()) {
                        if (!inventory.branchCode) {
                            ctx.addIssue({
                                code: zod_1.z.ZodIssueCode.custom,
                                path: [index, 'items', itemIndex, 'stockInventory', inventoryIndex, 'branchCode'],
                                message: 'Branch code is required when stock inventory is provided.',
                            });
                        }
                        if (inventory.itemStock == null) {
                            ctx.addIssue({
                                code: zod_1.z.ZodIssueCode.custom,
                                path: [index, 'items', itemIndex, 'stockInventory', inventoryIndex, 'itemStock'],
                                message: 'Item stock is required when stock inventory is provided.',
                            });
                        }
                    }
                }
            }
        }
    }
});
exports.itemMasterStatusSchema = zod_1.z.object({
    status: zod_1.z.string()
        .min(1, { message: "Status is required" })
        .max(1, { message: "Status must be a single character" })
        .refine(value => value === "1" || value === "2", {
        message: "Status must be either '1' or '2'"
    })
});
exports.itemDetailStatusSchema = zod_1.z.object({
    status: zod_1.z.string()
        .min(1, { message: "Status is required" })
        .max(1, { message: "Status must be a single character" })
        .refine(value => value === "1" || value === "2", {
        message: "Status must be either '1' or '2'"
    })
        .transform(value => value.trim()), // Optional: Trim whitespace from the status value
});
exports.itemStockUpdateSchema = zod_1.z.object({
    stockInventory: zod_1.z.array(zod_1.z.object({
        itemCode: zod_1.z.string().nonempty('Item code is required').refine(async (itemCode) => {
            const exists = await mongoose_1.default
                .model(collections_1.collections.inventoryManagement.items.itemsDetails)
                .countDocuments({ itemCode });
            return exists > 0;
        }, {
            message: 'Item code does not exist in the database.',
        }),
        branchCode: zod_1.z.string().min(1, { message: 'Branch code is required.' }),
        serialNumbers: zod_1.z.array(zod_1.z.string()).optional(),
        batchCodes: zod_1.z.array(zod_1.z.object({
            batchCode: zod_1.z.string().min(1, 'Batch code is required'),
            itemStock: zod_1.z
                .number()
                .transform((val) => Number(val))
                .refine((val) => !isNaN(val), 'Item Stock must be a valid number')
                .refine((val) => val >= 0, 'Item Stock must be positive'),
        })).optional(),
        transferBranches: zod_1.z.array(zod_1.z.object({
            transferBranchCode: zod_1.z.string().optional(),
            serialNumber: zod_1.z.string().optional(),
            batchCode: zod_1.z.string().optional(),
            itemStock: zod_1.z
                .number()
                .transform((val) => Number(val))
                .refine((val) => !isNaN(val), 'Item Stock must be a valid number')
                .refine((val) => val >= 0, 'Item Stock must be positive'),
        })).optional(),
        usedItems: zod_1.z.array(zod_1.z.object({
            serialNumber: zod_1.z.string().optional(),
            batchCode: zod_1.z.string().optional(),
            itemStock: zod_1.z
                .number()
                .transform((val) => Number(val))
                .refine((val) => !isNaN(val), 'Item Stock must be a valid number')
                .refine((val) => val >= 0, 'Item Stock must be positive'),
        })).optional(),
        itemStock: zod_1.z.number().optional(),
    })),
}).superRefine(async (data, ctx) => {
    const allSerialNumbers = [];
    const allBatchCodes = [];
    for (const stockItem of data.stockInventory) {
        if (stockItem.serialNumbers) {
            allSerialNumbers.push(...stockItem.serialNumbers);
        }
        if ((0, lodash_1.size)(stockItem.batchCodes) > 0) {
            allBatchCodes.push(...stockItem.batchCodes.map((batchCode) => batchCode?.batchCode));
        }
    }
    const duplicateSerialNumbers = allSerialNumbers.filter((value, idx, self) => self.indexOf(value) !== idx);
    if (duplicateSerialNumbers.length > 0) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: `Duplicate serial numbers found: ${duplicateSerialNumbers.join(', ')}`,
            path: ['serialNumbers'],
        });
    }
    const duplicateBatchCodes = allBatchCodes.filter((value, idx, self) => self.indexOf(value) !== idx);
    if (duplicateBatchCodes.length > 0) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: `Duplicate batch codes found: ${duplicateBatchCodes.join(', ')}`,
            path: ['batchCodes'],
        });
    }
    const existingSerialNumbers = await item_details_serial_number_1.default.find({
        serialNumber: { $in: allSerialNumbers },
    }, { serialNumber: 1 });
    if ((0, lodash_1.size)(existingSerialNumbers) > 0) {
        const existingSerialNumberList = existingSerialNumbers.map(item => item.serialNumber);
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: `The following serial numbers already exist in the database: ${existingSerialNumberList.join(', ')}`,
            path: ['serialNumbers'],
        });
    }
    const existingBatchCodes = await item_details_serial_number_1.default.find({
        batchNumber: { $in: allBatchCodes },
    }, { batchNumber: 1 });
    if ((0, lodash_1.size)(existingBatchCodes) > 0) {
        const existingBatchCodeList = existingBatchCodes.map(item => item.batchNumber);
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: `The following batch codes already exist in the database: ${existingBatchCodeList.join(', ')}`,
            path: ['batchCodes'],
        });
    }
});
exports.itemRequestAcceptFormSchema = zod_1.z.object({
    requestId: zod_1.z.string({ required_error: 'Request id is required', }).min(2, 'Request id is should be 2 chars minimum'),
    requestedBranchId: zod_1.z.string({ required_error: 'Requested branch is required', }).min(2, 'Requested branch is should be 2 chars minimum'),
    requestedStatus: zod_1.z.string({ required_error: 'Requested status is required', }).min(2, 'Requested status is should be 2 chars minimum'),
    batchSerial: zod_1.z.array(zod_1.z.object({
        serialNumber: zod_1.z.string({
            required_error: 'Serial number is required',
        }),
        quantity: zod_1.z
            .number({
            required_error: 'Quantity is required',
        }).optional(),
    })).optional(),
    serialNumbers: zod_1.z.array(zod_1.z.string()).optional(),
    acceptedQuantity: zod_1.z
        .number({
        required_error: 'Quantity is required',
    })
        .min(1, 'Quantity must not be empty')
        .transform((value) => Number(value))
        .refine((val) => !isNaN(val) && val > 0, {
        message: 'Quantity must be a valid positive number',
    }).optional(),
    requestedQuantity: zod_1.z
        .number({
        required_error: 'Quantity is required',
    })
        .min(1, 'Quantity must not be empty')
        .transform((value) => Number(value))
        .refine((val) => !isNaN(val) && val > 0, {
        message: 'Quantity must be a valid positive number',
    }),
});
