"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.productStockUpdateSchema = exports.productSchema = void 0;
const mongoose_1 = __importDefault(require("mongoose"));
const zod_1 = require("zod");
const collections_1 = require("../../../../../constants/collections");
const items_1 = require("../../../../../constants/items");
const lodash_1 = require("lodash");
const products_serial_number_1 = __importDefault(require("../../../../../model/admin/inventory-management/products/products-serial-number"));
const itemAttributeschema = zod_1.z.object({
    attributeId: zod_1.z.string({ required_error: 'Key column is required', }).min(1, { message: "Attribute ID is required" }),
    attributeDetailId: zod_1.z.string({ required_error: 'Key column is required', }).min(1, { message: "Attribute Detail ID is required" }),
    attributeName: zod_1.z.string({ required_error: 'Key column is required', }).min(1, { message: "Attribute name is required" }),
    itemValue: zod_1.z.string({ required_error: 'Key column is required', }).min(1, { message: "Item value is required" }),
});
exports.productSchema = zod_1.z.array(zod_1.z.object({
    _id: zod_1.z.string().optional(),
    categoryId: zod_1.z.string().refine((id) => mongoose_1.default.Types.ObjectId.isValid(id), {
        message: 'Invalid category ID',
    }),
    warrantyId: zod_1.z.string().optional().nullable(),
    itemTitle: zod_1.z.string().min(1, { message: 'Item title is required.' }),
    itemCode: zod_1.z.string().min(3, { message: 'Item code must be at least 3 characters long.' }),
    itemType: zod_1.z.enum([items_1.itemTypes.normal, items_1.itemTypes.bom]).optional(),
    itemDescription: zod_1.z.string().optional(),
    itemImageUrl: zod_1.z.string().url({ message: 'Item image URL must be a valid URL.' }).optional(),
    barCode: zod_1.z.string().optional(),
    hsnCode: zod_1.z.string().optional(),
    unit: zod_1.z.string().optional(),
    measurements: zod_1.z.object({
        weight: zod_1.z.string().optional(),
        height: zod_1.z.string().optional(),
        length: zod_1.z.string().optional(),
        width: zod_1.z.string().optional(),
    }).optional(),
    itemAttributes: zod_1.z.any(),
    status: zod_1.z.enum(['1', '2', '3']).optional(),
})).superRefine(async (dataArray, ctx) => {
    // const itemCodeSet = new Set<string>();
    // const itemCodes = dataArray.map(data => data.itemCode);
    // const existingItems = await mongoose.model(collections.inventoryManagement.products.products).find({ itemCode: { $in: itemCodes } }).select('itemCode _id').lean();
    // const existingCodeMap = existingItems.reduce((acc: any, item: any) => {
    //     acc[item.itemCode] = toString(item._id);
    //     return acc;
    // }, {});
    for (const [index, data] of dataArray.entries()) {
        const { _id, itemCode, itemAttributes } = data;
        const id = _id ? new mongoose_1.default.Types.ObjectId(_id) : undefined;
        // if (has(itemCodeSet, itemCode)) {
        //     ctx.addIssue({
        //         code: zod.ZodIssueCode.custom,
        //         path: [index, 'itemCode'],
        //         message: `Item code '${itemCode}' is duplicated at index ${index}.`,
        //     });
        // } else {
        //     itemCodeSet.add(itemCode);
        // }
        // const existingItemId = existingCodeMap[itemCode];
        // if (existingItemId && (!id || existingItemId !== toString(id))) {
        //     ctx.addIssue({
        //         code: zod.ZodIssueCode.custom,
        //         path: [index, 'itemCode'],
        //         message: 'Item code must be unique.',
        //     });
        // }
        if (!(0, lodash_1.isArray)(itemAttributes)) {
            ctx.addIssue({
                code: zod_1.z.ZodIssueCode.custom,
                path: [index, 'itemAttributes'],
                message: 'itemAttributes must be an array of attributes',
            });
            continue;
        }
        const globalAttributeComboSet = new Set();
        for (const [index, data] of dataArray.entries()) {
            const { itemAttributes } = data;
            if (!(0, lodash_1.isArray)(itemAttributes))
                continue;
            const attributeComboSet = new Set();
            const attributeNameSet = new Set();
            const attributeIdSet = new Set();
            const attributeDetailIdSet = new Set();
            const itemValueSet = new Set();
            for (let attrIndex = 0; attrIndex < itemAttributes.length; attrIndex++) {
                const attribute = itemAttributes[attrIndex];
                const result = itemAttributeschema.safeParse(attribute);
                if (!result.success) {
                    (0, lodash_1.forEach)(result.error.issues, issue => {
                        ctx.addIssue({
                            code: issue.code,
                            path: [index, 'itemAttributes', attrIndex, ...issue.path],
                            message: issue.message,
                        });
                    });
                }
                else {
                    const attributeNameKey = attribute.attributeName;
                    if (attributeIdSet.has(attribute.attributeId)) {
                        ctx.addIssue({
                            code: zod_1.z.ZodIssueCode.custom,
                            path: [index, 'itemAttributes', attrIndex, 'attributeId'],
                            message: `Duplicate attributeId '${attribute.attributeId}' found in itemAttributes. Must be unique.`,
                        });
                    }
                    else {
                        attributeIdSet.add(attribute.attributeId);
                    }
                    if (attributeNameSet.has(attributeNameKey)) {
                        ctx.addIssue({
                            code: zod_1.z.ZodIssueCode.custom,
                            path: [index, 'itemAttributes', attrIndex, 'attributeName'],
                            message: `Duplicate attributeName '${attributeNameKey}' found in itemAttributes. All attribute names must be unique.`,
                        });
                    }
                    else {
                        attributeNameSet.add(attributeNameKey);
                    }
                    if (attributeDetailIdSet.has(attribute.attributeDetailId)) {
                        ctx.addIssue({
                            code: zod_1.z.ZodIssueCode.custom,
                            path: [index, 'itemAttributes', attrIndex, 'attributeDetailId'],
                            message: `Duplicate attributeDetailId '${attribute.attributeDetailId}' found in itemAttributes. Must be unique.`,
                        });
                    }
                    else {
                        attributeDetailIdSet.add(attribute.attributeDetailId);
                    }
                    if (itemValueSet.has(attribute.itemValue)) {
                        ctx.addIssue({
                            code: zod_1.z.ZodIssueCode.custom,
                            path: [index, 'itemAttributes', attrIndex, 'itemValue'],
                            message: `Duplicate itemValue '${attribute.itemValue}' found in itemAttributes. Must be unique.`,
                        });
                    }
                    else {
                        itemValueSet.add(attribute.itemValue);
                    }
                    const comboKey = [attribute.attributeId, attribute.attributeDetailId, attribute.attributeName, attribute.itemValue].join('|');
                    if (attributeComboSet.has(comboKey)) {
                        ctx.addIssue({
                            code: zod_1.z.ZodIssueCode.custom,
                            path: [index, 'itemAttributes', attrIndex],
                            message: `Duplicate attribute combination within this product.`,
                        });
                    }
                    else {
                        attributeComboSet.add(comboKey);
                    }
                    if (globalAttributeComboSet.has(comboKey)) {
                        ctx.addIssue({
                            code: zod_1.z.ZodIssueCode.custom,
                            path: [index, 'itemAttributes', attrIndex],
                            message: `Duplicate attribute combination found across multiple products.`,
                        });
                    }
                    else {
                        globalAttributeComboSet.add(comboKey);
                    }
                }
            }
        }
    }
});
exports.productStockUpdateSchema = zod_1.z.object({
    stockInventory: zod_1.z.array(zod_1.z.object({
        itemCode: zod_1.z.string().nonempty('Item code is required').refine(async (itemCode) => {
            const exists = await mongoose_1.default
                .model(collections_1.collections.inventoryManagement.products.products)
                .countDocuments({ itemCode });
            return exists > 0;
        }, {
            message: 'Item code does not exist in the database.',
        }),
        branchCode: zod_1.z.string().min(1, { message: 'Branch code is required.' }),
        serialNumbers: zod_1.z.array(zod_1.z.string()).optional(),
        batchCodes: zod_1.z.array(zod_1.z.object({
            batchCode: zod_1.z.string().min(1, 'Batch code is required'),
            itemStock: zod_1.z
                .number()
                .transform((val) => Number(val))
                .refine((val) => !isNaN(val), 'Item Stock must be a valid number')
                .refine((val) => val >= 0, 'Item Stock must be positive'),
        })).optional(),
        transferBranches: zod_1.z.array(zod_1.z.object({
            transferBranchCode: zod_1.z.string().optional(),
            serialNumber: zod_1.z.string().optional(),
            batchCode: zod_1.z.string().optional(),
            itemStock: zod_1.z
                .number()
                .transform((val) => Number(val))
                .refine((val) => !isNaN(val), 'Item Stock must be a valid number')
                .refine((val) => val >= 0, 'Item Stock must be positive'),
        })).optional(),
        usedItems: zod_1.z.array(zod_1.z.object({
            serialNumber: zod_1.z.string().optional(),
            batchCode: zod_1.z.string().optional(),
            itemStock: zod_1.z
                .number()
                .transform((val) => Number(val))
                .refine((val) => !isNaN(val), 'Item Stock must be a valid number')
                .refine((val) => val >= 0, 'Item Stock must be positive'),
        })).optional(),
        itemStock: zod_1.z.number().optional(),
    })),
}).superRefine(async (data, ctx) => {
    const allSerialNumbers = [];
    const allBatchCodes = [];
    for (const stockItem of data.stockInventory) {
        if (stockItem.serialNumbers) {
            allSerialNumbers.push(...stockItem.serialNumbers);
        }
        if ((0, lodash_1.size)(stockItem.batchCodes) > 0) {
            allBatchCodes.push(...stockItem.batchCodes.map((batchCode) => batchCode?.batchCode));
        }
    }
    const duplicateSerialNumbers = allSerialNumbers.filter((value, idx, self) => self.indexOf(value) !== idx);
    if (duplicateSerialNumbers.length > 0) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: `Duplicate serial numbers found: ${duplicateSerialNumbers.join(', ')}`,
            path: ['serialNumbers'],
        });
    }
    const duplicateBatchCodes = allBatchCodes.filter((value, idx, self) => self.indexOf(value) !== idx);
    if (duplicateBatchCodes.length > 0) {
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: `Duplicate batch codes found: ${duplicateBatchCodes.join(', ')}`,
            path: ['batchCodes'],
        });
    }
    const existingSerialNumbers = await products_serial_number_1.default.find({
        serialNumber: { $in: allSerialNumbers },
    }, { serialNumber: 1 });
    if ((0, lodash_1.size)(existingSerialNumbers) > 0) {
        const existingSerialNumberList = existingSerialNumbers.map(item => item.serialNumber);
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: `The following serial numbers already exist in the database: ${existingSerialNumberList.join(', ')}`,
            path: ['serialNumbers'],
        });
    }
    const existingBatchCodes = await products_serial_number_1.default.find({
        batchNumber: { $in: allBatchCodes },
    }, { batchNumber: 1 });
    if ((0, lodash_1.size)(existingBatchCodes) > 0) {
        const existingBatchCodeList = existingBatchCodes.map(item => item.batchNumber);
        ctx.addIssue({
            code: zod_1.z.ZodIssueCode.custom,
            message: `The following batch codes already exist in the database: ${existingBatchCodeList.join(', ')}`,
            path: ['batchCodes'],
        });
    }
});
